<?php

namespace App\Http\Controllers\backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Notification;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $datas=User::where('status','N')->where('role','Administrator')->get();
        return view('backend.user.index',compact('datas'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.user.add');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validateData=$request->validate([
            'email'=>'unique:users|required',
            'f_name'=>'required|min:2',
            'l_name'=>'required|min:3',
            'phone' => 'required|numeric',
            'password' => 'min:8',
            'confirm_password' => 'required_with:password|same:password|min:8',
        ]);
        $input=array();
        $input['f_name']=ucwords($request->f_name);
        $input['l_name']=ucwords($request->l_name);
        $input['email']=$request->email;
        $input['phone']=$request->phone;
        $input['role']='Administrator';
        $input['is_email_verified']=1;
        $input['password']=bcrypt($request->password);
        $result = User::create($input);
        if($result){
            //notification
            $notification=new Notification();
            $notification->title='Administrator Sign up';
            $notification->description=$request->f_name.' Registered as Administrator <br>Email:'.$request->email;
            $notification->save();

            $notification=array(
                'message'=>'User Save Successfully!!',
                'alert-type'=>'success'
            );
            return redirect()->route('user.view')->with($notification);
        }else{
            $notification=array(
                'message'=>'Some Problem on server!!',
                'alert-type'=>'error'
            );
            return redirect()->back()->with($notification);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $data=User::find($id);
        return view('backend.user.show',compact('data'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $data=User::find($id);
        return view('backend.user.edit',compact('data'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validateData=$request->validate([
            'f_name'=>'required|min:2',
            'l_name'=>'required|min:3',
            'phone' => 'required|numeric',
        ]);

        $data= User::find($id);
        $data->f_name=ucwords($request->f_name);
        $data->l_name=ucwords($request->l_name);
        $data->phone=$request->phone;
        $data->updated_by=Auth::user()->id;
        $result=$data->save();
        if($result){
            $notification=array(
                'message'=>'User Save Successfully!!',
                'alert-type'=>'success'
            );
            return redirect()->route('user.view')->with($notification);
        }else{
            $notification=array(
                'message'=>'Some Problem on server!!',
                'alert-type'=>'error'
            );
            return redirect()->back()->with($notification);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $data=User::find($id);
        $data->deleted_by=Auth::user()->id;
        $data->status='Y';
        $data->deleted_datetime=date('Y-m-d H:i:s');
        $result=$data->update();
        $notification=array(
            'message'=>'User Delete Successfully!!',
            'alert-type'=>'info'
        );
        return redirect()->route('user.view')->with($notification);
    }
    public function ChangePassword()
    {
        return view('backend.user.change_password');
    }
    //ChangePasswordSave
    public function ChangePasswordSave(Request $request)
    {
        $validateData=$request->validate([
            'old_password' => 'min:8 |required',
            'password' => 'min:8',
            'confirm_password' => 'required_with:password|same:password|min:8'
        ]);

        #Match The Old Password
        if(!Hash::check($request->old_password, auth()->user()->password)){
            $notification=array(
                'message'=>"Old Password Doesn't match!",
                'alert-type'=>'error'
            );
            return redirect()->back()->with($notification);
        }


        #Update the new Password
        auth()->user()->update([
            'password' => Hash::make($request->password),
        ]);

        $notification=array(
            'message'=>'Password changed successfully!',
            'alert-type'=>'success'
        );
        return redirect()->back()->with($notification);
    }



    public function inActiveUser()
    {
        $datas=User::where('status','Y')->where('role','Administrator')->get();
        return view('backend.user.inactive',compact('datas'));
    }

    public function inActiveUserEdit($id)
    {
        $data=User::find($id);
        $data->updated_by=Auth::user()->id;
        $data->status='N';
        $result=$data->update();
        $notification=array(
            'message'=>'User Active Successfully!!',
            'alert-type'=>'info'
        );
        return redirect()->route('user.inactive.view')->with($notification);
    }
}
