<?php

namespace App\Http\Controllers\frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Notification;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Mail;
use App\Models\EmailManagement;
use App\Models\StateList;
use App\Models\UserVerify;
use App\Models\PartnerNotification;
use Illuminate\Support\Str;
use App\Providers\MessageManagementProvider;
use App\Providers\EmailProvider;
use Illuminate\Support\Facades\Log;
use App\Rules\PartnerCodeCheck;
use stdClass;


class CustomerController extends Controller
{
    public function create(){
        if(Auth::check()){
            return redirect()->route('dashboard');
        }else{
            $state_list = StateList::all();
            return view('frontend.new_user.customer_registration',compact('state_list'));
        }
    }

    public function store(Request $request){
        $validateData = $request->validate([
            'email' => 'unique:users|required',
            'f_name' => 'required|min:2',
            'l_name' => 'required|min:2',
            'company' => 'required|min:2',
            'country' => 'required|min:2',
            'calling_code' => 'required',
            'phone' => 'required|numeric',
            'pincode' => strtolower($request->country) === 'ae' ? 'nullable' : 'required|numeric',
            'statelist' => strtolower($request->country) === 'in' ? 'required' : 'nullable',
            'password' => ['required', 'min:8', 'regex:/^(?=.*?[A-Z])(?=.*?[a-z])(?=.*?[0-9])(?=.*?[#?!@$%^&*-]).{8,}$/'],
            'password_confirmation' => ['required', 'min:8', 'same:password'],
            'partner_code' => ['nullable', new PartnerCodeCheck],
        ], [
            'f_name.required' => 'Please enter first name.',
            'f_name.min' => 'First name must be greater than 2 characters long.',
            'l_name.required' => 'Please enter last name.',
            'l_name.min' => 'Last name must be greater than 2 characters long.',
            'email.unique' => 'User already exists. You can use <a href="'.route('forget.password.get').'">forget password</a> function or register with a different email address.',
            'email.required' => 'Please enter an email address.',
            'company.required' => 'Please enter company name.',
            'country.required' => 'Please enter country name.',
            'phone.required' => 'Please enter a phone number.',
            'password.required' => 'Please enter a password.',
            'pincode.required' => 'Please enter a pincode.',
            'calling_code.required' => 'Please select a calling code.',
            'statelist.required' => 'Please select a state.',
            'partner_code.required' => 'Please enter Partner code.',
        ]);
        
        try{
            //-------------unique code generate------------------
            $year = date("Y");
            $code=$year.'01';
            $fname=strtoupper(substr(trim($request->company),0,2));
            $gCode=$fname.$code;
            $rowCount=User::where('code',$gCode)->count();
            while($rowCount!=0){
                $code++;
                $gCode=$fname.$code;
                $rowCount=User::where('code',$gCode)->count();
            }

            $input=array();
            $input['code']=$gCode;
            $input['partner_code']=$request->partner_code;
            $input['f_name']=ucwords($request->f_name);
            $input['l_name']=ucwords($request->l_name);
            $input['company']=$request->company;
            $input['country']=$request->country;
            $input['state']=($request->country=='IN')?$request->statelist:Null;
            $input['city']=$request->city;
            $input['address']=$request->address;
            $input['pincode']=($request->country=='AE')?Null:$request->pincode;
            $input['email']=$request->email;
            $input['calling_code']=$request->calling_code;
            $input['phone']=$request->phone;
            $input['role']='Customer';
            $input['password']=bcrypt($request->password);
            $result = User::create($input);

            if(@$result->id){
                //notification
                $notification=new Notification();
                $notification->notification_from= '1';
                $notification->table_id= $result->id;
                $notification->title=$request->f_name. ' '.$request->l_name.' has registered as a customer';
                $notification->description=$request->f_name.' Registered as customer <br> <b>Code:</b>'.$gCode. ' <b>Email:</b>'.$request->email;
                $notification->save();

                //partner_code partner Notification
                if($request->partner_code){
                    $partnerDetail=User::where('code',$request->partner_code)->first();
                    if($partnerDetail){
                        //notification partner
                        $partner_notification=new PartnerNotification();
                        $partner_notification->user_id =$partnerDetail->id;
                        $partner_notification->notification_from ='1';
                        $partner_notification->table_id =$result->id;
                        $partner_notification->notification='New Customer'.$request->f_name. ' '.$request->l_name.'( '.$gCode.' ) Registered';
                        $partner_notification->save();
                    }
                }

                $token = Str::random(64);
                $link=route('user.verify', ['token'=>$token,'email'=>$request->email]);
                $number=rand(10000,99999);
                UserVerify::create([
                    'user_id' => $result->id,
                    'token' => $token,
                    'otp' => $number
                ]);
                $emailList=EmailManagement::first();
                //otp mail
                $value=new stdClass();
                $value->f_name=$request->f_name;
                $value->l_name=$request->l_name;
                $value->otp=$number;
                EmailProvider::Email('customer_registration_otp', $request->email, $value );

                $email=$request->email;
                $success_message=MessageManagementProvider::MessageManage('registration-success-message');
                return view('frontend.new_user.otp',compact('token','email','success_message'));
            }else{
                return redirect()->back();
            }
        }catch(\Exception $e) {
            if(@$result->id){
                User::find(@$result->id)->delete();
                UserVerify::where('user_id',$result->id)->delete();
            }
            if(@$notification->id){
                Notification::find(@$notification->id)->delete();
            }
            Log::error('Customer Register Error: '. $e);
            $message = MessageManagementProvider::MessageManage('domain-may-not-exist');
            return redirect()->back()->with('message',$message);
        }
    }

    public function verifyAccount($token,$email){
        try{
            $userList=User::where('email',$email)->where('status','N')->first();
            $verifyUser = UserVerify::where('token', $token)->first();
            $message =MessageManagementProvider::MessageManage('email-not-exist');
            if(!is_null($verifyUser) && @$userList->id==@$verifyUser->user_id ){
                $user = $verifyUser->user;

                if(!$user->is_email_verified) {
                    return view('frontend.new_user.otp',compact('token','email'));
                } else {
                    $message =MessageManagementProvider::MessageManage('email-verified');
                }
            }
            return redirect()->route('home')->with('message', $message);
        }catch(\Exception $e){
            Log::error('email verifide'.$e);
            $message =MessageManagementProvider::MessageManage('excetion-error');
            return redirect()->route('home')->with('message', $message);
        }
    }

    //registration otp
    public function verifyOtpAccount(Request $request){
        $validateData=$request->validate([
            'token' => 'required',
            'otp' => 'required'
        ]);
        try{
            $token=$request->token;
            $verifyUser = UserVerify::where('token', $token)->where('otp', $request->otp)->first();

            $message =MessageManagementProvider::MessageManage('email-not-exist');
            if(!is_null($verifyUser) ){
                $user = $verifyUser->user;

                if(!$user->is_email_verified) {
                    $secondsDifference=strtotime(date('Y-m-d H:i:s'))-strtotime($verifyUser->updated_at);
                    if(round($secondsDifference/60)<=10){
                        $verifyUser->user->is_email_verified = 1;
                        $verifyUser->user->save();
                        //welcome mail;
                        if($user->role=='Partner'){
                            EmailProvider::Email('become_a_partner_registration_welcome', $user->email, $user );
                            $support_team_email_list=EmailManagement::where('key','support_team_email')->first();
                            EmailProvider::Email('become_a_partner_registration_welcome_for_support', $support_team_email_list->title, $user );
                        }else{
                            EmailProvider::Email('customer_welcome', $user->email, $user );
                        }

                        $message =MessageManagementProvider::MessageManage('email-verify');
                        return redirect()->route('home')->with('message', $message);
                    }else{
                        $message =MessageManagementProvider::MessageManage('otp-expired');
                        $message1 =MessageManagementProvider::MessageManage('email-again');
                        return view('frontend.new_user.otp')->with(['message'=> $message, 'message1'=>$message1,'email'=>$user->email,'token'=>$token]);
                    }
                } else {
                    $message =MessageManagementProvider::MessageManage('email-verified');
                    return redirect()->route('home')->with('message', $message);
                }
            }
            $verifyUser1 = UserVerify::where('token', $token)->first();
            $user = @$verifyUser1->user;
            if(@$user){
                if($user->status=='N'){
                    $message =MessageManagementProvider::MessageManage('wrong-otp');
                    return view('frontend.new_user.otp',compact('message','token'));
                }else{
                    $message =MessageManagementProvider::MessageManage('account-delete-email-verify');
                    return view('frontend.new_user.otp',compact('message','token'));
                }
            }
            return view('frontend.new_user.otp')->with(['message'=> $message,'token'=>$token,'email'=>@$user->email]);
        }catch(\Exception $e) {
            Log::error('otp email verification Error: '. $e);
            $message=MessageManagementProvider::MessageManage('domain-may-not-exist');
            return redirect()->back()->with('message',$message);
        }
    }

    public function ResendOtp(){
       return view('frontend.new_user.reSend_otp');
    }

    public function ResendOtpEmail(Request $request){

        $validateData=$request->validate([
            'email' => 'required|email'
        ]);
        try{
            $user=User::where('email',$request->email)->where('is_email_verified',0)->where('status','N')->first();
            if(!is_null($user)){
                $verifyUser=UserVerify::where('user_id',$user->id)->delete();
                $token = Str::random(64);
                $number=rand(10000,99999);
                $link=route('user.verify', ['token'=>$token,'email'=>$request->email]);
                UserVerify::create([
                    'user_id' => $user->id,
                    'token' => $token,
                    'otp' => $number
                ]);

                //otp mail
                $value=new stdClass();
                $value->f_name=$user->f_name;
                $value->l_name=$user->l_name;
                $value->otp=$number;
                if($user->role=='Partner'){
                    EmailProvider::Email('become_a_partner_registration_forgot_password', $request->email, $value );
                }else{
                    EmailProvider::Email('forgot_password', $request->email, $value );
                }

                $email=$request->email;
                return view('frontend.new_user.otp',compact('token','email'));
            }else{
                $user=User::where('email',$request->email)->first();
                if($user){
                    if($user->status=='N'){
                        $message =MessageManagementProvider::MessageManage('email-verified');
                        return view('frontend.new_user.reSend_otp',compact('message'));
                    }else{
                        $message =MessageManagementProvider::MessageManage('account-delete-email-verify');
                        return view('frontend.new_user.reSend_otp',compact('message'));
                    }
                }
                $message =MessageManagementProvider::MessageManage('registred-mail-email-verify');
                return view('frontend.new_user.reSend_otp',compact('message'));
            }
        }catch(\Exception $e){
            Log::error('Resend Otp Email varification '.$e);
            $message =MessageManagementProvider::MessageManage('excetion-error');
            return redirect()->back()->with('message',$message);
        }
    }



    //registration otp
    public function verifyOtpAccountAjax(Request $request){
        try{
            $email=$request->email;
            $token=$request->token;
            $verifyUser = UserVerify::where('token', $token)->where('otp', $request->otp)->first();

            if(!is_null($verifyUser) ){
                $user = $verifyUser->user;

                if(!$user->is_email_verified) {
                    $secondsDifference=strtotime(date('Y-m-d H:i:s'))-strtotime($verifyUser->updated_at);
                    if(round($secondsDifference/60)<=10){
                        $verifyUser->user->is_email_verified = 1;
                        $verifyUser->user->save();
                        $message['otp'] =MessageManagementProvider::MessageManage('email-verify');
                        $message['login'] =true;
                        $message['email']=$request->email;
                        //welcome mail;
                        if($user->role=='Partner'){
                            EmailProvider::Email('become_a_partner_registration_welcome', $request->email, $user );
                            $support_team_email_list=EmailManagement::where('key','support_team_email')->first();
                            EmailProvider::Email('become_a_partner_registration_welcome_for_support', $support_team_email_list->title, $user );
                        }else{
                            EmailProvider::Email('customer_welcome', $request->email, $user );
                        }
                        return $message;
                    }else{
                        $message['otp'] =MessageManagementProvider::MessageManage('otp-expired');
                        $message['email']=$request->email;
                        $message['token']=$request->token;
                        return $message;
                    }
                } else {
                    $message['otp'] =MessageManagementProvider::MessageManage('email-verified');
                    $message['login']=true;
                    $message['email']=$request->email;
                    return $message;
                }
            }
            $verifyUser1 = UserVerify::where('token', $token)->first();
            $user = @$verifyUser1->user;
            if(@$user){
                if($user->status=='N'){
                    $message['otp'] =MessageManagementProvider::MessageManage('wrong-otp');
                    $message['email']=$request->email;
                    $message['token']=$request->token;
                    return $message;
                }else{
                    $message['otp'] =MessageManagementProvider::MessageManage('account-delete-email-verify');
                    $message['login'] =true;
                    $message['email']=$request->email;
                    return $message;
                }
            }
            $message['otp'] =MessageManagementProvider::MessageManage('email-not-exist');
            $message['login'] =true;
            $message['email']=$request->email;
            return $message;
        }catch(\Exception $e){
            $message =MessageManagementProvider::MessageManage('excetion-error');
            $message['otp'] =$message;
            $message['email']=$request->email;
            return $message;
        }
    }

    public function ResendOtpEmailAjax(Request $request){
        try{
            $user=User::where('email',$request->email)->where('is_email_verified',0)->where('status','N')->first();
            if(!is_null($user)){
                $verifyUser=UserVerify::where('user_id',$user->id)->delete();
                $token = Str::random(64);
                $number=rand(10000,99999);
                UserVerify::where('user_id',$user->id)->delete();
                UserVerify::create([
                    'user_id' => $user->id,
                    'token' => $token,
                    'otp' => $number
                ]);

                //otp mail
                $value=new stdClass();
                $value->f_name=$user->f_name;
                $value->l_name=$user->l_name;
                $value->otp=$number;
                if($user->role=='Partner'){
                    EmailProvider::Email('become_a_partner_registration_registration_otp', $request->email, $value );
                }else{
                    EmailProvider::Email('customer_registration_otp', $request->email, $value );
                }

                $email=$request->email;
                $message['otp']=MessageManagementProvider::MessageManage('resend-otp');
                $message['email']=$email;
                $message['token']=$token;
                return $message;
            }else{
                $user=User::where('email',$request->email)->first();
                if($user){
                    if($user->status=='N'){
                        $message['otp'] =MessageManagementProvider::MessageManage('email-verified');
                        $message['login'] =true;
                        return $message;
                    }else{
                        $message['otp'] =MessageManagementProvider::MessageManage('account-delete-email-verify');
                        $message['login'] =true;
                        return $message;
                    }
                }
                $message['otp'] =MessageManagementProvider::MessageManage('registred-mail-email-verify');
                $message['login'] =true;
                return $message;
            }
        }catch(\Exception $e){
            $message['otp'] =MessageManagementProvider::MessageManage('domain-may-not-exist');
            return $message;
        }
    }

    public function ResendOtpEmailCheckAjax(Request $request){
        try{
            $user=User::where('email',$request->email)->where('is_email_verified',0)->where('status','N')->first();
            $message['login']=false;
            if(!is_null($user)){
                $message['otp']=MessageManagementProvider::MessageManage("validate-email-popup");
                $message['login']=true;
                return $message;
            }else{
                $user=User::where('email',$request->email)->first();
                if($user){
                    if($user->status=='N'){
                        $message['otp'] =MessageManagementProvider::MessageManage('email-verified');
                        return $message;
                    }else{
                        $message['otp'] =MessageManagementProvider::MessageManage('account-delete-email-verify');
                        return $message;
                    }
                }
                $message['otp'] =MessageManagementProvider::MessageManage('registred-mail-email-verify');
                return $message;
            }
        }catch(\Exception $e){
            $message['otp'] =$e;
            return $message;
        }
    }

    public function CheckCustomerEmail(Request $request){
        $emailCheck=User::where('email',$request->email)->count();
        if($emailCheck>0){
            return true;
        }else{
            return false;
        }
    }

    public function CheckPartnerCode(Request $request){
        $currentDate=date('Y-m-d');
        $emailCheck=User::where('code',$request->partner_code)->where('role','Partner')->where('status','N')->where('is_email_verified','1')->where('agreement_to_date','>',$currentDate)->count();
        if($emailCheck>0){
            return true;
        }
        return false;
    }


}
