<?php
/**
 * RedParts sputnik ajax.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\AJAX' ) ) {
	/**
	 * Class AJAX.
	 */
	class AJAX extends Singleton {
		/**
		 * Initialization.
		 */
		public function init() {
			if ( ! wp_doing_ajax() ) {
				return;
			}

			add_filter( 'redparts_sputnik_ajax_widget_settings', array( $this, 'set_group_index' ), 10, 2 );

			add_action( 'wp_ajax_redparts_sputnik_widget', array( $this, 'widget' ) );
			add_action( 'wp_ajax_nopriv_redparts_sputnik_widget', array( $this, 'widget' ) );
		}

		/**
		 * Outputs widget.
		 */
		public function widget() {
			if (
				! isset( $_POST['nonce'] ) ||
				! wp_verify_nonce( sanitize_key( wp_unslash( $_POST['nonce'] ) ), 'redparts_sputnik_widget' )
			) {
				wp_die( esc_html__( 'Action failed. Please refresh the page and retry.', 'redparts-sputnik' ) );
			}

			WPML::switch_ajax_language();

			if (
				! isset( $_POST['data']['widget_name'] ) ||
				! isset( $_POST['data']['widget_type'] ) ||
				! isset( $_POST['data']['widget_id'] )
			) {
				wp_die( -1 );
			}

			$widget_name = sanitize_key( wp_unslash( $_POST['data']['widget_name'] ) );
			$widget_type = sanitize_key( wp_unslash( $_POST['data']['widget_type'] ) );

			$allowed_widgets = array(
				'redparts_sputnik_block_products_carousel' => '\RedParts\Sputnik\Widgets\Block_Products_Carousel',
				'redparts_sputnik_block_zone'              => '\RedParts\Sputnik\Widgets\Block_Zone',
			);

			if ( ! isset( $allowed_widgets[ $widget_name ] ) ) {
				wp_die( -1 );
			}

			$widget_class = $allowed_widgets[ $widget_name ];

			if ( ! class_exists( $widget_class ) ) {
				wp_die( -1 );
			}

			$settings = array();
			$widget   = new $widget_class();

			if ( 'default' === $widget_type ) {
				$widget_id = absint( wp_unslash( $_POST['data']['widget_id'] ) );
				$settings  = $widget->get_settings();

				if ( ! isset( $settings[ $widget_id ] ) ) {
					wp_die();
				}

				$settings = $settings[ $widget_id ];
			} elseif ( 'elementor' === $widget_type ) {
				if (
					! isset( $_POST['data']['elementor_post_id'] ) ||
					! isset( $_POST['data']['elementor_widget_id'] )
				) {
					wp_die( -1 );
				}

				$post_id   = absint( wp_unslash( $_POST['data']['elementor_post_id'] ) );
				$widget_id = sanitize_key( wp_unslash( $_POST['data']['elementor_widget_id'] ) );
				$post_meta = get_post_meta( $post_id, '_elementor_data', true );

				if ( ! $post_meta ) {
					wp_die( -1 );
				}

				$post_meta = json_decode( $post_meta, true );

				/**
				 * Returns elementor widget.
				 *
				 * @param array  $document  - Elementor page content.
				 * @param string $widget_id - Elementor widget ID.
				 *
				 * @return array|null
				 * @noinspection PhpMissingReturnTypeInspection
				 */
				function get_elementor_widget( array $document, string $widget_id ) {
					foreach ( $document as $widget ) {
						if ( isset( $widget['id'] ) && $widget_id === $widget['id'] ) {
							return $widget;
						}
						if ( ! empty( $widget['elements'] ) ) {
							$widget = get_elementor_widget( $widget['elements'], $widget_id );

							if ( $widget ) {
								return $widget;
							}
						}
					}

					return null;
				}

				$elementor_widget = get_elementor_widget( $post_meta, $widget_id );

				if ( ! $elementor_widget || ! isset( $elementor_widget['settings']['wp'] ) ) {
					wp_die( -1 );
				}

				$settings = $elementor_widget['settings']['wp'];
			} else {
				wp_die( -1 );
			}

			$settings = apply_filters( 'redparts_sputnik_ajax_widget_settings', $settings, $widget_name );

			$widget->widget( array(), $settings );

			wp_die();
		}

		/**
		 * Sets group index to the widget settings.
		 *
		 * @param array  $settings    Widget settings array.
		 * @param string $widget_name Widget name.
		 * @return array
		 */
		public function set_group_index( array $settings, string $widget_name ): array {
			$allowed_widgets = array(
				'redparts_sputnik_block_products_carousel',
				'redparts_sputnik_block_zone',
			);

			if (
				! in_array( $widget_name, $allowed_widgets, true ) ||
				! isset( $_POST['nonce'] ) ||
				! wp_verify_nonce( sanitize_key( wp_unslash( $_POST['nonce'] ) ), 'redparts_sputnik_widget' ) ||
				! isset( $_POST['data']['group_index'] )
			) {
				return $settings;
			}

			$group_index = absint( wp_unslash( $_POST['data']['group_index'] ) );
			$groups      = isset( $settings['groups'] ) && is_array( $settings['groups'] )
				? array_values( $settings['groups'] )
				: array();

			if ( ! isset( $groups[ $group_index ] ) ) {
				return $settings;
			}

			$settings['current_group'] = $group_index;

			return $settings;
		}
	}
}
