<?php
/**
 * This file contains helpers to working with colors.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Colors' ) ) {
	/**
	 * Class Colors
	 */
	class Colors {
		/**
		 * Returns the color in RGB format, represented as an array.
		 *
		 * @param string $hex - Color in the HEX format. For example: #fff, #ffffff.
		 *
		 * @return array|null
		 * @noinspection PhpMissingReturnTypeInspection
		 */
		public static function hex_to_rgb( string $hex ) {
			if ( ! preg_match( '/^#(([A-Fa-f0-9]{3}){1,2})$/', $hex ) ) {
				return null;
			}

			$hex = substr( $hex, 1 );

			if ( 3 === strlen( $hex ) ) {
				$rgb = array_map(
					function ( $part ) {
						return hexdec( $part . $part );
					},
					str_split( $hex )
				);
			} else {
				$rgb = array_map(
					function ( $part ) {
						return hexdec( $part );
					},
					str_split( $hex, 2 )
				);
			}

			return $rgb;
		}

		/**
		 * Returns color luminance.
		 *
		 * @param string $hex - Color in the HEX format. For example: #fff, #ffffff.
		 *
		 * @return float|null
		 * @noinspection PhpMissingReturnTypeInspection
		 */
		public static function luminance( string $hex ) {
			$rgb = self::hex_to_rgb( $hex );

			if ( ! $hex ) {
				return null;
			}

			$rgb = array_map(
				function( $part ) {
					$part = $part / 255;

					if ( $part <= 0.03928 ) {
						return $part / 12.92;
					} else {
						return pow( ( $part + 0.055 ) / 1.055, 2.4 );
					}
				},
				$rgb
			);

			return 0.2126 * $rgb[0] + 0.7152 * $rgb[1] + 0.0722 * $rgb[2];
		}

		/**
		 * Compares to colors and return contrast value.
		 *
		 * @param string $hex1 - Color in the HEX format. For example: #fff, #ffffff.
		 * @param string $hex2 - Color in the HEX format. For example: #fff, #ffffff.
		 *
		 * @return float
		 */
		public static function contrast( string $hex1, string $hex2 ) {
			$l1 = self::luminance( $hex1 );
			$l2 = self::luminance( $hex2 );

			if ( null === $l1 || null === $l2 ) {
				return null;
			}

			return ( max( $l1, $l2 ) + 0.05 ) / ( min( $l1, $l2 ) + 0.05 );
		}

		/**
		 * Returns type of color. One of [white, light, dark, black].
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @param string $hex - Color in the HEX format. For example: #fff, #ffffff.
		 *
		 * @return string
		 */
		public static function type( string $hex ): string {
			static $cache = array();

			if ( ! isset( $cache[ $hex ] ) ) {
				$white_contrast = self::contrast( $hex, '#fff' );
				$black_contrast = self::contrast( $hex, '#000' );

				if ( 1.0 === $white_contrast && 21.0 === $black_contrast ) {
					$result = 'white';
				} elseif ( 21.0 === $white_contrast && 1.0 === $black_contrast ) {
					$result = 'black';
				} elseif ( $white_contrast >= 3.0 && $black_contrast < 10.0 ) {
					$result = 'dark';
				} else {
					$result = 'light';
				}

				$cache[ $hex ] = $result;
			}

			return $cache[ $hex ];
		}
	}
}
