<?php
/**
 * RedParts share buttons.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 *
 * @noinspection DuplicatedCode
 */

namespace RedParts\Sputnik;

use WP_Post;
use WP_Term;
use stdClass;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Share_Buttons' ) ) {
	/**
	 * Class Share_Buttons
	 */
	class Share_Buttons extends Singleton {
		/**
		 * Array of available share buttons.
		 *
		 * @var array
		 */
		private $available_buttons = array();

		/**
		 * Initialization.
		 */
		public function init() {
			$this->available_buttons = array(
				'facebook'  => array(
					'name'  => esc_html__( 'Facebook', 'redparts-sputnik' ),
					'label' => esc_html__( 'Like', 'redparts-sputnik' ),
					'url'   => 'https://www.facebook.com/sharer/sharer.php?u=%URL%',
					'icon'  => 'fa fa-facebook',
					'color' => '#4267b2',
				),
				'twitter'   => array(
					'name'  => esc_html__( 'Twitter', 'redparts-sputnik' ),
					'label' => esc_html__( 'Tweet', 'redparts-sputnik' ),
					'url'   => 'https://twitter.com/share?url=%URL%',
					'icon'  => 'fa fa-twitter',
					'color' => '#1b95e0',
				),
				'pinterest' => array(
					'name'  => esc_html__( 'Pinterest', 'redparts-sputnik' ),
					'label' => esc_html__( 'Pin It', 'redparts-sputnik' ),
					'url'   => 'https://pinterest.com/pin/create/button/?url=%URL%&media=%IMAGE%&description=%TITLE%',
					'icon'  => 'fa fa-pinterest',
					'color' => '#e63939',
				),
				'whatsapp'  => array(
					'name'  => esc_html__( 'Whatsapp', 'redparts-sputnik' ),
					'label' => esc_html__( 'Share', 'redparts-sputnik' ),
					'url'   => 'https://wa.me/?text=%URL%',
					'icon'  => 'fa fa-whatsapp',
					'color' => '#00e676',
				),
				'linkedin'  => array(
					'name'  => esc_html__( 'LinkedIn', 'redparts-sputnik' ),
					'label' => esc_html__( 'Share', 'redparts-sputnik' ),
					'url'   => 'https://www.linkedin.com/shareArticle?title=%TITLE%&url=%URL%',
					'icon'  => 'fa fa-linkedin',
					'color' => '#006599',
				),
				'ok'        => array(
					'name'  => esc_html__( 'Odnoklassniki', 'redparts-sputnik' ),
					'label' => esc_html__( 'Share', 'redparts-sputnik' ),
					'url'   => 'https://connect.ok.ru/offer?url=%URL%&title=%TITLE%&imageUrl=%IMAGE%',
					'icon'  => 'fa fa-odnoklassniki',
					'color' => '#eb7815',
				),
				'vk'        => array(
					'name'  => esc_html__( 'Vkontakte', 'redparts-sputnik' ),
					'label' => esc_html__( 'Share', 'redparts-sputnik' ),
					'url'   => 'https://vk.com/share.php?url=%URL%',
					'icon'  => 'fa fa-vk',
					'color' => '#4a76a8',
				),
				'telegram'  => array(
					'name'  => esc_html__( 'Telegram', 'redparts-sputnik' ),
					'label' => esc_html__( 'Share', 'redparts-sputnik' ),
					'url'   => 'https://telegram.me/share/url?url=%URL%&text=%TITLE%',
					'icon'  => 'fa fa-paper-plane',
					'color' => '#2ba0da',
				),
				'email'     => array(
					'name'  => esc_html__( 'Email', 'redparts-sputnik' ),
					'label' => esc_html__( 'Share', 'redparts-sputnik' ),
					'url'   => 'mailto:?to=&subject=%TITLE%&body=%URL%',
					'icon'  => 'fa fa-envelope',
					'color' => '#404040',
				),
			);

			add_action( 'admin_head-nav-menus.php', array( $this, 'init_nav_menu_meta_box' ) );
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
			add_filter( 'walker_nav_menu_start_el', array( $this, 'nav_menu_item_icon' ), 10, 4 );
			add_filter( 'nav_menu_css_class', array( $this, 'nav_menu_item_classes' ), 10, 4 );
			add_filter( 'nav_menu_link_attributes', array( $this, 'nav_menu_item_href' ), 10, 3 );
			add_shortcode( 'redparts_sputnik_share_buttons', array( $this, 'shortcode' ) );
		}

		/**
		 * Adds a nav menu meta box.
		 */
		public function init_nav_menu_meta_box() {
			add_meta_box(
				'redparts_sputnik_share_buttons',
				esc_html__( 'RedParts Share Buttons', 'redparts-sputnik' ),
				array( $this, 'nav_menu_meta_box' ),
				'nav-menus',
				'side',
				'high'
			);
		}

		/**
		 * Enqueue scripts.
		 */
		public function enqueue_scripts() {
			wp_add_inline_style( 'redparts-sputnik-style', $this->render_css() );
		}

		/**
		 * Returns available share buttons.
		 *
		 * @return array
		 */
		public function get_available_buttons(): array {
			return apply_filters( 'redparts_sputnik_get_available_share_buttons', $this->available_buttons );
		}

		/**
		 * Renders and prints share buttons HTML.
		 *
		 * @param string $classes CSS classes of the root HTML element.
		 */
		public function render( string $classes = '' ) {
			$menu_id = $this->get_menu_id();

			ob_start();

			$menu = wp_get_nav_menu_object( $menu_id );

			if ( $menu ) {
				wp_nav_menu(
					array(
						'container_class' => $classes . ' th-share-buttons',
						'menu'            => $menu,
						'depth'           => 1,
						'link_before'     => '<span class="menu-item__label">',
						'link_after'      => '</span>',
						'fallback_cb'     => '__return_empty_string',
					)
				);
			}

			// Because we have already escaped the output above.
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo apply_filters( 'redparts_sputnik_share_buttons_render', ob_get_clean(), $classes );
		}

		/**
		 * Renders and returns share buttons CSS.
		 *
		 * @return string
		 */
		public function render_css(): string {
			$buttons = $this->get_available_buttons();
			$result  = '';

			foreach ( $buttons as $key => $button ) {
				$result .= '.th-share-buttons .th-social-' . sanitize_key( $key ) . ' a';
				$result .= '{background: ' . sanitize_hex_color( $button['color'] ) . ';}';
			}

			return apply_filters( 'redparts_sputnik_share_buttons_render_css', $result );
		}

		/**
		 * Share buttons shortcode.
		 *
		 * @param array $attrs - Shortcode attributes.
		 *
		 * @return string
		 */
		public function shortcode( array $attrs ): string {
			$attrs = shortcode_atts(
				array(
					'classes' => '',
				),
				$attrs
			);

			ob_start();

			$this->render( $attrs['classes'] );

			return ob_get_clean();
		}

		/**
		 * Adds icon to the nav menu item.
		 *
		 * @noinspection PhpUnusedParameterInspection,PhpMissingParamTypeInspection
		 *
		 * @param string   $output - The menu item output.
		 * @param WP_Post  $item   - Menu item object.
		 * @param int      $depth  - Depth of the menu.
		 * @param stdClass $args   - wp_nav_menu() arguments.
		 *
		 * @return string
		 */
		public function nav_menu_item_icon( string $output, $item, int $depth, stdClass $args ): string {
			static $added_styles = array();

			if ( ! isset( $args->menu ) ) {
				return $output;
			}

			$menu_id = $this->get_menu_id();

			if ( $menu_id === $args->menu || ( $args->menu instanceof WP_Term && $menu_id === $args->menu->term_id ) ) {
				$buttons = $this->get_available_buttons();
				/** Nav menu item has classes property. @noinspection PhpUndefinedFieldInspection */
				$classes = explode( ' ', implode( ' ', $item->classes ) );
				$icons   = array();
				$type    = '';
				$color   = '';

				foreach ( $classes as $class ) {
					if ( preg_match( '#^th-icon-(.+)$#', $class, $mr ) ) {
						$icons[] = $mr[1];
					}
					if ( preg_match( '#^th-social-(.+)$#', $class, $mr ) ) {
						$type = $mr[1];
					}
					if ( preg_match( '#^th-color-((?:[A-Fa-f0-9]{3}){1,2})$#', $class, $mr ) ) {
						$color = $mr[1];
					}
				}

				if ( $type && empty( $icons ) && isset( $buttons[ $type ] ) ) {
					$icons = explode( ' ', $buttons[ $type ]['icon'] );
				}

				$icons  = implode( ' ', $icons );
				$icons  = $icons ? $icons : 'fa fa-link';
				$output = str_replace(
					$args->link_before,
					'<i class="menu-item__icon ' . esc_attr( $icons ) . '"></i> ' . $args->link_before,
					$output
				);

				if ( $color ) {
					$style_id = $menu_id . '-' . $item->ID;

					if ( ! isset( $added_styles[ $style_id ] ) ) {
						$added_styles[ $style_id ] = true;

						$style = '.th-share-buttons .menu-item-' . $item->ID . ' a {background: #' . $color . '}';

						wp_add_inline_style( 'redparts-sputnik-inline', $style );
					}

					if ( is_customize_preview() ) {
						$output = str_replace( '<a', '<a style="background: ' . sanitize_hex_color( '#' . $color ) . '"', $output );
					}
				}
			}

			return $output;
		}

		/**
		 * Returns menu item classes.
		 *
		 * @noinspection PhpMissingParamTypeInspection
		 *
		 * @param string[] $classes Array of the CSS classes that are applied to the menu item's `<li>` element.
		 * @param WP_Post  $item    The current menu item.
		 * @param stdClass $args    An object of wp_nav_menu() arguments.
		 * @param int      $depth   Depth of menu item. Used for padding.
		 *
		 * @return string[]
		 */
		public function nav_menu_item_classes( array $classes, $item, stdClass $args, int $depth ): array {
			if ( ! isset( $args->menu ) ) {
				return $classes;
			}

			$menu_id = $this->get_menu_id();

			if ( $menu_id === $args->menu || ( $args->menu instanceof WP_Term && $menu_id === $args->menu->term_id ) ) {
				/** Nav menu item has title property. @noinspection PhpUndefinedFieldInspection */
				$title = apply_filters( 'the_title', $item->title, $item->ID );
				$title = apply_filters( 'nav_menu_item_title', $title, $item, $args, $depth );

				if ( empty( $title ) ) {
					$classes[] = 'menu-item--without-label';
				}
			}

			return $classes;
		}

		/**
		 * Modifies menu item href.
		 *
		 * @noinspection PhpUnusedParameterInspection,PhpMissingParamTypeInspection
		 *
		 * @param array    $attrs The HTML attributes applied to the menu item's `<a>` element.
		 * @param WP_Post  $item  The current menu item.
		 * @param stdClass $args  An object of wp_nav_menu() arguments.
		 *
		 * @return array
		 */
		public function nav_menu_item_href( array $attrs, $item, stdClass $args ): array {
			if ( ! isset( $args->menu ) ) {
				return $attrs;
			}

			$menu_id = $this->get_menu_id();

			if ( $menu_id === $args->menu || ( $args->menu instanceof WP_Term && $menu_id === $args->menu->term_id ) ) {
				$attrs['href'] = str_replace(
					array(
						'%URL%',
						'%IMAGE%',
						'%TITLE%',
					),
					array(
						rawurlencode( get_permalink() ),
						rawurlencode( get_the_post_thumbnail_url( null, 'full' ) ),
						rawurlencode( get_the_title() ),
					),
					$attrs['href']
				);
			}

			return $attrs;
		}

		/**
		 * Outputs nav menu meta box.
		 */
		public function nav_menu_meta_box() {
			$buttons = $this->get_available_buttons();
			$box_id  = 'redparts-sputnik-share-buttons';

			?>
			<div id="post-type-<?php echo esc_attr( $box_id ); ?>" class="posttypediv">
				<div id="tabs-panel-<?php echo esc_attr( $box_id ); ?>" class="tabs-panel tabs-panel-active">
					<ul id="<?php echo esc_attr( $box_id ); ?>-checklist" class="categorychecklist form-no-clear">
						<?php
						$i = -1;
						foreach ( $buttons as $key => $button ) :
							?>
							<li>
								<label class="menu-item-title">
									<input
										type="checkbox"
										class="menu-item-checkbox"
										name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-object-id]"
										value="<?php echo esc_attr( $i ); ?>"
									/> <?php echo esc_html( $button['name'] ); ?>
								</label>
								<input
									type="hidden"
									class="menu-item-type"
									name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-type]"
									value="custom"
								/>
								<input
									type="hidden"
									class="menu-item-title"
									name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-title]"
									value="<?php echo esc_attr( $button['label'] ); ?>"
								/>
								<input
									type="hidden"
									class="menu-item-url"
									name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-url]"
									value="<?php echo esc_attr( $button['url'] ); ?>"
								/>
								<input
									type="hidden"
									class="menu-item-classes"
									name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-classes]"
									value="<?php echo esc_attr( 'th-social-' . $key ); ?>"
								/>
								<input
									type="hidden"
									class="menu-item-target"
									name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-target]"
									value="_blank"
								/>
							</li>
							<?php
							$i--;
						endforeach;
						?>
					</ul>
				</div>
				<p class="button-controls">
					<span class="list-controls">
						<a
							href="<?php echo esc_url( admin_url( 'nav-menus.php?page-tab=all&selectall=1#post-type-' . $box_id ) ); ?>"
							class="select-all"
						>
							<?php esc_html_e( 'Select all', 'redparts-sputnik' ); ?>
						</a>
					</span>
					<span class="add-to-menu">
						<button
							type="submit"
							class="button-secondary submit-add-to-menu right"
							value="<?php esc_attr_e( 'Add to menu', 'redparts-sputnik' ); ?>"
							name="add-post-type-menu-item"
							id="submit-post-type-<?php echo esc_attr( $box_id ); ?>"
						>
							<?php esc_html_e( 'Add to menu', 'redparts-sputnik' ); ?>
						</button>
						<span class="spinner"></span>
					</span>
				</p>
			</div>
			<?php
		}

		/**
		 * Returns ID of share buttons menu.
		 *
		 * @since 1.6.0
		 *
		 * @return int
		 */
		public function get_menu_id(): int {
			$menu_id = absint( Settings::instance()->get( 'share_buttons_menu' ) );

			return absint( apply_filters( 'wpml_object_id', $menu_id, 'nav_menu', true ) );
		}
	}
}
