<?php
/**
 * This file contains code related to the wishlist.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;
use WC_Product;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Wishlist' ) ) {
	/**
	 * Class Wishlist
	 */
	class Wishlist extends Singleton {
		const COOKIE_KEY = 'redparts_sputnik_wishlist';

		/**
		 * List of product ids in wishlist.
		 *
		 * @var array
		 */
		protected $products_list = array();

		/**
		 * Initialization.
		 */
		public function init() {
			add_action( 'init', array( $this, 'deferred_init' ) );
		}

		/**
		 * Deferred initialization.
		 *
		 * @since 1.5.0
		 */
		public function deferred_init() {
			if ( ! $this->is_enabled() ) {
				return;
			}

			add_action( 'redparts_sputnik_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

			add_shortcode( 'redparts_sputnik_wishlist', array( $this, 'shortcode' ) );

			if ( wp_doing_ajax() ) {
				add_action( 'wp_ajax_redparts_sputnik_wishlist_add', array( $this, 'ajax' ) );
				add_action( 'wp_ajax_nopriv_redparts_sputnik_wishlist_add', array( $this, 'ajax' ) );

				add_action( 'wp_ajax_redparts_sputnik_wishlist_remove', array( $this, 'ajax' ) );
				add_action( 'wp_ajax_nopriv_redparts_sputnik_wishlist_remove', array( $this, 'ajax' ) );
			}

			// Button content.
			add_action( 'redparts_sputnik_wishlist_button_content', array( $this, 'the_button_label' ), 100 );
			add_action( 'redparts_sputnik_wishlist_button_content', array( $this, 'the_button_icon' ), 200 );

			// Initializes the product list.
			if ( is_user_logged_in() ) {
				$products_list = get_user_meta( get_current_user_id(), self::COOKIE_KEY, true );

				if ( ! empty( $products_list ) ) {
					$this->products_list = $products_list;
				}
			} elseif ( isset( $_COOKIE[ self::COOKIE_KEY ] ) ) {
				$this->products_list = wp_parse_id_list( wp_unslash( $_COOKIE[ self::COOKIE_KEY ] ) );
			}
		}

		/**
		 * Returns true if compare module enabled
		 *
		 * @since 1.5.0
		 *
		 * @return bool
		 */
		public function is_enabled(): bool {
			return 'no' !== Settings::instance()->get( 'wishlist_enabled', 'yes' );
		}

		/**
		 * Enqueue scripts.
		 */
		public function enqueue_scripts() {
			wp_register_script(
				'redparts-sputnik-wishlist',
				Plugin::instance()->url( 'assets/js/wishlist.js' ),
				array( 'jquery' ),
				Plugin::VERSION,
				true
			);
			wp_localize_script(
				'redparts-sputnik-wishlist',
				'redPartsSputnikWishlistVars',
				array(
					'ajaxUrl'     => apply_filters( 'redparts_sputnik_ajax_url', '' ),
					'pageUrl'     => $this->get_page_url(),
					'nonceAdd'    => wp_create_nonce( 'redparts_sputnik_wishlist_add' ),
					'nonceRemove' => wp_create_nonce( 'redparts_sputnik_wishlist_remove' ),
					'button'      => array(
						'added' => array(
							'label' => apply_filters(
								'redparts_sputnik_wishlist_button_added_label',
								esc_html__( 'Go to wishlist', 'redparts-sputnik' )
							),
							'icon'  => apply_filters(
								'redparts_sputnik_wishlist_button_added_icon',
								''
							),
						),
					),
				)
			);

			wp_enqueue_script( 'redparts-sputnik-wishlist' );
		}

		/**
		 * Returns number of products in wish list.
		 *
		 * @return int
		 */
		public function count(): int {
			return count( $this->products_list );
		}

		/**
		 * Shortcode to display wish list.
		 */
		public function shortcode() {
			global $product;

			$get_translated_product_id = function( $product_id ) {
				return apply_filters( 'wpml_object_id', $product_id, 'product', true );
			};

			/** Products array. @var WC_Product[] $products */
			$products = array_map( 'wc_get_product', array_map( $get_translated_product_id, $this->products_list ) );

			ob_start();

			?>
			<div class="th-block th-block-wishlist">
				<?php if ( $products ) : ?>
					<div class="th-container">
						<div class="th-wishlist">
							<table class="th-wishlist__table">
								<thead class="th-wishlist__head">
								<tr class="th-wishlist__row wishlist__row--head">
									<th class="th-wishlist__column th-wishlist__column--head th-wishlist__column--image">
										<?php esc_html_e( 'Image', 'redparts-sputnik' ); ?>
									</th>
									<th class="th-wishlist__column th-wishlist__column--head th-wishlist__column--product">
										<?php esc_html_e( 'Product', 'redparts-sputnik' ); ?>
									</th>
									<th class="th-wishlist__column th-wishlist__column--head th-wishlist__column--stock">
										<?php esc_html_e( 'Stock status', 'redparts-sputnik' ); ?>
									</th>
									<th class="th-wishlist__column th-wishlist__column--head th-wishlist__column--price">
										<?php esc_html_e( 'Price', 'redparts-sputnik' ); ?>
									</th>
									<th class="th-wishlist__column th-wishlist__column--head th-wishlist__column--button"></th>
									<th class="th-wishlist__column th-wishlist__column--head th-wishlist__column--remove"></th>
								</tr>
								</thead>
								<tbody class="th-wishlist__body">
								<?php foreach ( $products as $product ) : ?>
									<tr class="th-wishlist__row wishlist__row--body">
										<td class="th-wishlist__column th-wishlist__column--body th-wishlist__column--image">
											<a href="<?php echo esc_url( $product->get_permalink() ); ?>">
												<?php echo wp_kses( $product->get_image(), 'redparts_sputnik_image' ); ?>
											</a>
										</td>
										<td class="th-wishlist__column th-wishlist__column--body th-wishlist__column--product">
											<div class="th-wishlist__product-name">
												<a href="<?php echo esc_url( $product->get_permalink() ); ?>">
													<?php echo esc_html( $product->get_title() ); ?>
												</a>
											</div>
											<?php if ( $product->get_rating_count() ) : ?>
												<?php $rating = $product->get_rating_count(); ?>
												<div class="th-wishlist__product-rating">
													<div class="th-wishlist__product-rating-stars">
														<?php wc_get_template( 'loop/rating.php' ); ?>
													</div>
													<div class="th-wishlist__product-rating-title">
														<?php
														echo esc_html(
															sprintf(
																// translators: %s: Number of reviews.
																_n( '%s Review', '%s Reviews', $rating, 'redparts-sputnik' ), // SKIP-ESC.
																$rating
															)
														);
														?>
													</div>
												</div>
											<?php endif; ?>
										</td>
										<td class="th-wishlist__column th-wishlist__column--body th-wishlist__column--stock">
											<?php $stock_html = wc_get_stock_html( $product ); ?>
											<?php if ( $stock_html ) : ?>
												<?php echo wp_kses( $stock_html, 'redparts_sputnik_stock' ); ?>
											<?php else : ?>
												<div class="th-wishlist__dash"></div>
											<?php endif; ?>
										</td>
										<td class="th-wishlist__column th-wishlist__column--body th-wishlist__column--price">
											<?php woocommerce_template_loop_price(); ?>
										</td>
										<td class="th-wishlist__column th-wishlist__column--body th-wishlist__column--button">
											<?php woocommerce_template_loop_add_to_cart(); ?>
										</td>
										<td class="th-wishlist__column th-wishlist__column--body th-wishlist__column--remove">
											<a
												class="th-wishlist__remove th-wishlist-remove th-button th-button--style--muted th-button--size--small th-button--icon"
												href="<?php echo esc_url( $this->get_remove_url( $product->get_id() ) ); ?>"
												data-product-id="<?php echo esc_attr( $product->get_id() ); ?>"
											>
												<?php
												// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
												echo apply_filters(
													'redparts_sputnik_wishlist_remove_button_content',
													esc_html__( 'Remove', 'redparts-sputnik' )
												);
												?>
											</a>
										</td>
									</tr>
								<?php endforeach; ?>
								</tbody>
							</table>
						</div>
					</div>
				<?php else : ?>
					<div class="th-block-empty th-block-empty--after-header">
						<div class="th-container">
							<div class="th-block-empty__body">
								<div class="th-block-empty__message">
									<?php
									echo wp_kses(
										__( 'You have no products in your wishlist.<br>Go to the home page to start browsing our store.', 'redparts-sputnik' ), // SKIP-ESC.
										'redparts_sputnik_text'
									);
									?>
								</div>
								<div class="th-block-empty__action">
									<?php
									$shop_url = apply_filters( 'woocommerce_return_to_shop_redirect', wc_get_page_permalink( 'shop' ) );
									?>
									<a href="<?php echo esc_url( $shop_url ); ?>" class="th-button th-button--style--primary th-button--size--small">
										<?php esc_html_e( 'Return to shop', 'redparts-sputnik' ); ?>
									</a>
								</div>
							</div>
						</div>
					</div>
				<?php endif; ?>
			</div>
			<?php

			return ob_get_clean();
		}

		/**
		 * Returns the number of products in the wish list.
		 *
		 * @return int
		 */
		public function get_count(): int {
			return count( $this->products_list );
		}

		/**
		 * Outputs the button to adding to the wish list.
		 *
		 * @noinspection PhpUnused
		 *
		 * @param integer|bool $product_id Product ID.
		 * @param string|array $classes    The list of classes that will be applied to the button.
		 */
		public function the_button( $product_id = false, $classes = '' ) {
			if ( ! $product_id ) {
				global $product;

				$product_id = ! is_null( $product ) && $product instanceof WC_Product ? $product->get_id() : 0;
			}

			if ( empty( $product_id ) ) {
				return;
			}

			$classes = apply_filters(
				'redparts_sputnik_wishlist_button_classes',
				redparts_sputnik_get_classes( 'th-wishlist-add', $classes )
			);

			ob_start();
			?>
			<a
				class="<?php echo esc_attr( implode( ' ', (array) $classes ) ); ?>"
				href="<?php echo esc_url( $this->get_add_url( $product_id ) ); ?>"
				data-product-id="<?php echo esc_attr( $product_id ); ?>"
			>
				<?php
				/**
				 * Hook: redparts_sputnik_wishlist_button_content.
				 *
				 * @hooked Wishlist::the_button_label - 100
				 * @hooked Wishlist::the_button_icon  - 200
				 */
				do_action( 'redparts_sputnik_wishlist_button_content' );
				?>
			</a>
			<?php

			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo apply_filters( 'redparts_sputnik_wishlist_the_button', ob_get_clean(), $product_id, $classes );
		}

		/**
		 * Outputs button label.
		 *
		 * @noinspection PhpUnused
		 */
		public function the_button_label() {
			?>
			<span class="th-wishlist-add__label">
				<?php
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo apply_filters(
					'redparts_sputnik_wishlist_button_label',
					esc_html__( 'Add to wishlist', 'redparts-sputnik' )
				);
				?>
			</span>
			<?php
		}

		/**
		 * Outputs button icon.
		 *
		 * @noinspection PhpUnused
		 */
		public function the_button_icon() {
			?>
			<span class="th-wishlist-add__icon">
				<?php
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo apply_filters( 'redparts_sputnik_wishlist_button_icon', '' );
				?>
			</span>
			<?php
		}

		/**
		 * Returns the URL to remove the product from the wish list.
		 *
		 * @param int $product_id The identifier of the product to be deleted.
		 * @return string
		 */
		public function get_remove_url( int $product_id ): string {
			$args = array(
				'action'     => 'remove',
				'product_id' => $product_id,
			);

			$url = $this->get_page_url();

			if ( ! empty( $url ) ) {
				$url = esc_url_raw( add_query_arg( $args, $this->get_page_url() ) );
			}

			return apply_filters( 'redparts_sputnik_wishlist_get_remove_url', $url, $product_id );
		}

		/**
		 * Returns the URL to add to the wish list.
		 *
		 * @param int $product_id Product ID.
		 * @return mixed|void
		 */
		public function get_add_url( int $product_id ) {
			$args = array(
				'action'     => 'add',
				'product_id' => $product_id,
			);

			$url = $this->get_page_url();

			if ( ! empty( $url ) ) {
				$url = esc_url_raw( add_query_arg( $args, $this->get_page_url() ) );
			}

			return apply_filters( 'redparts_sputnik_wishlist_get_add_url', $url, $product_id );
		}

		/**
		 * Returns wish list page URL.
		 *
		 * @return string
		 */
		public function get_page_url(): string {
			$page_id = Settings::instance()->get( 'wishlist_page' );
			$url     = $page_id ? get_page_link( $page_id ) : '';

			return apply_filters( 'redparts_sputnik_wishlist_get_page_url', $url );
		}

		/**
		 * Handles AJAX requests to add or remove products from the wish list.
		 *
		 * @noinspection DuplicatedCode
		 */
		public function ajax() {
			if ( ! isset( $_POST['nonce'] ) || ! isset( $_POST['action'] ) || ! isset( $_POST['product_id'] ) ) {
				wp_send_json_error();
				wp_die();
			}

			WPML::switch_ajax_language();

			$nonce      = sanitize_key( wp_unslash( $_POST['nonce'] ) );
			$action     = sanitize_key( wp_unslash( $_POST['action'] ) );
			$product_id = WPML::get_original_product_id( absint( wp_unslash( $_POST['product_id'] ) ) );

			if ( ! wp_verify_nonce( $nonce, $action ) ) {
				wp_send_json_error();
				wp_die();
			}

			if ( 'redparts_sputnik_wishlist_add' === $action ) {
				$product = wc_get_product( $product_id );

				if ( ! $product ) {
					wp_send_json_error();
					wp_die();
				}

				if ( ! in_array( $product_id, $this->products_list, true ) ) {
					$this->products_list[] = $product_id;
				}
			} elseif ( 'redparts_sputnik_wishlist_remove' === $action ) {
				if ( in_array( $product_id, $this->products_list, true ) ) {
					$this->products_list = array_filter(
						$this->products_list,
						function( $item ) use ( $product_id ) {
							return $item !== $product_id;
						}
					);
				}
			} else {
				wp_send_json_error();
				wp_die();
			}

			if ( is_user_logged_in() ) {
				update_user_meta( get_current_user_id(), self::COOKIE_KEY, $this->products_list );
			} else {
				setcookie( self::COOKIE_KEY, implode( ',', $this->products_list ), 0, COOKIEPATH, COOKIE_DOMAIN, false, false );
			}

			$fragments = apply_filters(
				'redparts_sputnik_wishlist_fragments',
				array(
					'.th-block-wishlist' => $this->shortcode(),
				),
				$action,
				$product_id
			);

			wp_send_json(
				apply_filters(
					'redparts_sputnik_wishlist_ajax_response',
					array(
						'success'   => true,
						'count'     => count( $this->products_list ),
						'fragments' => $fragments,
					),
					$action,
					$product_id
				)
			);
			wp_die();
		}
	}
}
