/*
 * Field RedPart Vehicle Import Export
 */

/*global redux, Papa, Base64 */

( function( $ ) {
	'use strict';

	redux.field_objects = redux.field_objects || {};
	redux.field_objects.redparts_vehicle_import_export = redux.field_objects.redparts_vehicle_import_export || {};

	redux.field_objects.redparts_vehicle_import_export.init = function( selector ) {
		selector = $.redux.getSelector( selector, 'redparts_vehicle_import_export' );

		$( selector ).each(
			function() {
				const el = $( this );
				let parent = el;

				if ( ! el.hasClass( 'redux-field-container' ) ) {
					parent = el.parents( '.redux-field-container:first' );
				}

				if ( parent.is( ':hidden' ) ) {
					return;
				}

				if ( parent.hasClass( 'redux-field-init' ) ) {
					parent.removeClass( 'redux-field-init' );
				} else {
					return;
				}

				let canceledRef = { current: false };
				let importCanceledRef = { current: false };

				$( '.redparts-export__include-compatibility', el ).on( 'change', function( event ) {
					event.stopPropagation();
				} );
				$( '.redparts-export__button-start', el ).on( 'click', function() {
					$( '.redparts-export', el ).attr( 'data-state', 'progress' );
					$( '.redparts-export .redparts-progress-bar__value', el ).css( 'width', '0%' );
					$( '.redparts-export .redparts-progress-bar__text', el ).text( '' );

					const ajaxUrl = $( this ).data( 'ajax-url' );
					const nonce = $( this ).data( 'nonce' );
					const limit = parseFloat( $( '.redparts-export', el ).attr( 'data-limit' ) );

					const includeCompatibility = $( '.redparts-export__include-compatibility', el )[ 0 ].checked;

					canceledRef = exportRequest( {
						ajaxUrl: ajaxUrl,
						nonce: nonce,
						compatibility: includeCompatibility ? '1' : '0',
						limit: limit,
					}, function( event ) {
						const completed = Math.round( ( event.completed / event.total ) * 100 );

						$( '.redparts-export .redparts-progress-bar__value', el ).css( 'width', completed + '%' );
						$( '.redparts-export .redparts-progress-bar__text', el ).text(
							event.completed + '/' + event.total
						);
					}, function( event ) {
						$( '.redparts-export', el ).attr( 'data-state', 'complete' );
						$( '.redparts-export__download-link', el ).prop( 'href', 'data:text/plain;base64,' + Base64.encode( event.csv ) );
					}, function( event ) {
						$( '.redparts-export', el ).attr( 'data-state', 'error' );
						$( '.redparts-export__error-message', el ).addClass( 'error' );
						$( '.redparts-export__error-message p', el ).text( event );
					} );
				} );
				$( '.redparts-export__button-cancel', el ).on( 'click', function() {
					$( '.redparts-export', el ).attr( 'data-state', 'start' );

					canceledRef.current = true;
				} );
				$( '.redparts-export__button-continue', el ).on( 'click', function() {
					$( '.redparts-export', el ).attr( 'data-state', 'start' );
				} );

				$( '.redparts-import__file', el ).on( 'change', function( event ) {
					event.stopPropagation();
				} );
				$( '.redparts-import__clear', el ).on( 'change', function( event ) {
					event.stopPropagation();
				} );
				$( '.redparts-import__button-start', el ).on( 'click', function() {
					const files = $( '.redparts-import__file', el )[ 0 ].files;
					const clearData = $( '.redparts-import__clear', el )[ 0 ].checked;
					const messages = $( '.redparts-import', el ).data( 'messages' );
					const limit = parseFloat( $( '.redparts-import', el ).attr( 'data-limit' ) );

					if ( 0 === files.length ) {
						// eslint-disable-next-line no-alert
						alert( messages.file_required );

						return;
					}

					$( '.redparts-import', el ).attr( 'data-state', 'progress' );
					$( '.redparts-import__log', el ).text( '' );

					const ajaxUrl = $( this ).data( 'ajax-url' );
					const nonce = $( this ).data( 'nonce' );
					const clearNonce = $( this ).data( 'clear-nonce' );

					let promise = Promise.resolve();

					if ( clearData ) {
						// eslint-disable-next-line no-alert
						if ( ! confirm( messages.confirm_clear ) ) {
							$( '.redparts-import', el ).attr( 'data-state', 'start' );
							return;
						}

						$( '.redparts-import__progress', el ).attr( 'data-type', 'clear' );
						$( '.redparts-import .redparts-progress-bar__value', el ).css( 'width', '0%' );
						$( '.redparts-import .redparts-progress-bar__text', el ).text( '' );

						const result = clearRequest( {
							ajaxUrl: ajaxUrl,
							nonce: clearNonce,
							onProgress: function( event ) {
								const completed = Math.round( ( event.completed / event.total ) * 100 );

								$( '.redparts-import .redparts-progress-bar__value', el ).css( 'width', completed + '%' );
								$( '.redparts-import .redparts-progress-bar__text', el ).text(
									event.completed + '/' + event.total
								);
							},
						} );

						importCanceledRef = result.canceledRef;
						promise = result.promise;
					}

					promise.then( function() {
						$( '.redparts-import__progress', el ).attr( 'data-type', 'import' );
						$( '.redparts-import .redparts-progress-bar__value', el ).css( 'width', '0%' );
						$( '.redparts-import .redparts-progress-bar__text', el ).text( '' );

						Papa.parse( files[ 0 ], {
							complete: function( results ) {
								const records = results.data;

								if ( 1 > records.length ) {
									return;
								}

								const lastIsEmpty = ! records[ records.length - 1 ].reduce( function( cell, acc ) {
									return acc && '' === cell;
								}, true );

								if ( lastIsEmpty ) {
									records.splice( -1 );
								}

								const header = records.splice( 0, 1 )[ 0 ];
								const rows = records.map( function( record ) {
									const row = {};

									for ( let i = 0; i < header.length; i++ ) {
										row[ header[ i ] ] = record[ i ];
									}

									return row;
								} );

								const chunks = [];

								while ( rows.length > 0 ) {
									chunks.push( rows.splice( 0, limit ) );
								}

								importCanceledRef = importRequest( {
									ajaxUrl: ajaxUrl,
									nonce: nonce,
									chunks: chunks,
									onProgress: function( event ) {
										const completed = Math.round( ( event.completed / event.total ) * 100 );

										$( '.redparts-import .redparts-progress-bar__value', el ).css( 'width', completed + '%' );
										$( '.redparts-import .redparts-progress-bar__text', el ).text(
											( event.completed * limit ) + '/' + ( event.total * limit )
										);

										event.errors.forEach( function( e ) {
											let text = $( '.redparts-import__log', el ).text();

											text += wp.i18n.sprintf( messages.log_message, e.row_number, e.message ) + '\n';

											$( '.redparts-import__log', el ).text( text );
										} );
									},
									onComplete: function() {
										$( '.redparts-import', el ).attr( 'data-state', 'complete' );
									},
									onError: function( event ) {
										$( '.redparts-import', el ).attr( 'data-state', 'error' );
										$( '.redparts-import__error-message', el ).addClass( 'error' );
										$( '.redparts-import__error-message p', el ).text( event );
									},
								} );
							},
						} );
					} );
				} );
				$( '.redparts-import__button-cancel', el ).on( 'click', function() {
					$( '.redparts-import', el ).attr( 'data-state', 'start' );

					importCanceledRef.current = true;
				} );
				$( '.redparts-import__button-continue', el ).on( 'click', function() {
					$( '.redparts-import', el ).attr( 'data-state', 'start' );
				} );
			}
		);
	};

	function exportRequest( options, onProgress, onComplete, onError ) {
		const canceledRef = { current: false };
		let result = '';
		let retries = 0;

		const request = function( limit, offset ) {
			retries++;

			if ( canceledRef.current ) {
				return;
			}

			const error = function( message ) {
				if ( 3 > retries ) {
					setTimeout( function() {
						request( limit, offset );
					}, 3000 );
				} else {
					onError( message );
				}
			};

			$.post( options.ajaxUrl, {
				action: 'redparts_sputnik_vehicles_export',
				nonce: options.nonce,
				data: {
					limit: limit,
					offset: offset,
					compatibility: options.compatibility,
				},
			}, function( response ) {
				if ( canceledRef.current ) {
					return;
				}

				if ( ! response.success ) {
					error( response.data.message );

					return;
				}

				retries = 0;

				const data = response.data;

				if ( '' === result ) {
					result += encodeCsvRow( data.header ) + '\n';
				}

				const part = data.rows.map( encodeCsvRow ).join( '\n' );

				result += part ? part + '\n' : '';

				onProgress( {
					completed: data.end,
					total: data.total,
				} );

				if ( data.total > data.end ) {
					request( limit, data.end );
				} else {
					onComplete( { csv: result } );
				}
			} ).fail( function( jqXHR ) {
				error( jqXHR.status + ' ' + jqXHR.statusText );
			} );
		};

		request( options.limit, 0 );

		return canceledRef;
	}

	function importRequest( options ) {
		const canceledRef = { current: false };
		const totalChunks = options.chunks.length;
		let retries = 0;
		let chunksCompleted = 0;
		let rowsCompleted = 0;

		const request = function( chunks ) {
			retries++;

			if ( canceledRef.current ) {
				return;
			}

			if ( 0 === chunks.length ) {
				options.onComplete();

				return;
			}

			const error = function( message ) {
				if ( 3 > retries ) {
					setTimeout( function() {
						request( chunks );
					}, 3000 );
				} else if ( options.onError ) {
					options.onError( message );
				}
			};

			$.post( options.ajaxUrl, {
				action: 'redparts_sputnik_vehicles_import',
				nonce: options.nonce,
				data: {
					rows: chunks[ 0 ],
				},
			}, function( response ) {
				if ( canceledRef.current ) {
					return;
				}

				if ( ! response.success ) {
					error( response.data.message );

					return;
				}

				retries = 0;

				const data = response.data;

				chunksCompleted += 1;
				rowsCompleted += chunks[ 0 ].length;

				options.onProgress( {
					completed: chunksCompleted,
					total: totalChunks,
					errors: data.errors.map( function( e ) {
						e.row_number = 2 + e.row_number + ( rowsCompleted - chunks[ 0 ].length );

						return e;
					} ),
				} );

				request( chunks.slice( 1 ) );
			} ).fail( function( jqXHR ) {
				error( jqXHR.status + ' ' + jqXHR.statusText );
			} );
		};

		request( options.chunks.slice() );

		return canceledRef;
	}

	function clearRequest( options ) {
		let retries = 0;
		let total = null;
		let remaining = null;
		const canceledRef = { current: false };
		const promise = new Promise( function( resolve, reject ) {
			const request = function() {
				retries++;

				if ( canceledRef.current ) {
					return;
				}

				if ( 0 === remaining ) {
					resolve();

					return;
				}

				const error = function( message ) {
					if ( 3 > retries ) {
						setTimeout( function() {
							request();
						}, 3000 );
					} else {
						reject( new Error( message ) );
					}
				};

				$.post( options.ajaxUrl, {
					action: 'redparts_sputnik_vehicles_clear',
					nonce: options.nonce,
				}, function( response ) {
					if ( canceledRef.current ) {
						return;
					}

					if ( ! response.success ) {
						error( response.data.message );

						return;
					}

					retries = 0;

					const data = response.data;

					remaining = data.remaining;

					if ( null === total ) {
						total = remaining + data.deleted;
					}

					options.onProgress( {
						total: total,
						completed: total - remaining,
					} );

					request();
				} ).fail( function( jqXHR ) {
					error( jqXHR.status + ' ' + jqXHR.statusText );
				} );
			};

			request();
		} );

		return {
			canceledRef: canceledRef,
			promise: promise,
		};
	}

	function encodeCsvRow( row ) {
		return row.map( function( cell ) {
			if ( cell.includes( '"' ) || cell.includes( ',' ) || cell.includes( '\r' ) || cell.includes( '\n' ) ) {
				return '"' + cell.replaceAll( '"', '""' ) + '"';
			}

			return cell;
		} ).join( ',' );
	}
}( jQuery ) );
