<?php
/**
 * Block brands widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;
use RedParts\Sputnik\Brands;
use WP_Term;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Brands' ) ) {
	/**
	 * Class Block_Products_Carousel
	 */
	class Block_Brands extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_brands',
				'description'                 => esc_html_x( 'Displays brands.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_brands',
				esc_html_x( 'RedParts: Block Brands', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$attribute_slug = Brands::instance()->get_attribute_slug();

			$without_images = 'yes' === ( $instance['without_images'] ?? 'no' );
			$limit          = max( 1, absint( $instance['limit'] ?? 16 ) );
			$brands         = isset( $instance['brands'] ) && is_string( $instance['brands'] ) ? $instance['brands'] : array();
			$brands         = get_terms(
				array(
					'taxonomy'   => $attribute_slug,
					'hide_empty' => false,
					'slug'       => wp_parse_slug_list( $brands ),
					'number'     => $limit,
					'orderby'    => 'slug__in',
				)
			);

			?>
			<!-- .th-block-brands -->
			<div class="th-block th-block-brands th-block-brands--layout--columns-8-full">
				<div class="th-container">
					<ul class="th-block-brands__list">
						<?php foreach ( $brands as $brand ) : ?>
							<?php
							/** WP_Term object. @var WP_Term $brand */

							$image = Brands::instance()->get_brand_image( $brand );
							$image = wp_get_attachment_image( $image, 'full' );

							if ( ! $image && ! $without_images ) {
								continue;
							}
							?>
							<li class="th-block-brands__item">
								<a
									href="<?php echo esc_url( get_term_link( $brand->term_id, $attribute_slug ) ); ?>"
									class="th-block-brands__item-link"
								>
									<?php echo wp_kses( $image, 'redparts_sputnik_image' ); ?>

									<span class="th-block-brands__item-name">
										<?php echo esc_html( $brand->name ); ?>
									</span>
								</a>
							</li>
							<li class="th-block-brands__divider" role="presentation"></li>
						<?php endforeach; ?>
					</ul>
				</div>
			</div>
			<!-- .th-block-brands / end -->
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'   => 'brands',
					'label' => esc_html__( 'Brands (comma separated slugs)', 'redparts-sputnik' ),
					'type'  => 'text',
				),
				array(
					'key'     => 'limit',
					'label'   => esc_html__( 'Maximum number of brands', 'redparts-sputnik' ),
					'type'    => 'number',
					'default' => '16',
				),
				array(
					'key'     => 'without_images',
					'label'   => esc_html__( 'Show brands without images', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'no',
				),
			);
		}

		/**
		 * Enqueue frontend scripts.
		 */
		public function enqueue_frontend_scripts() {
			wp_enqueue_script( 'redparts-sputnik-widget-block-brands' );
		}
	}
}
