<?php
/**
 * Block zone widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;
use WP_Query;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Zone' ) ) {
	/**
	 * Class Block_Zone
	 */
	class Block_Zone extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_block_zone',
				'description'                 => esc_html_x( 'Displays products from one of the main categories.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_zone',
				esc_html_x( 'RedParts: Block Zone', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return;
			}

			$category             = isset( $instance['category'] ) ? $instance['category'] : '';
			$subcategories        = isset( $instance['subcategories'] ) ? wp_parse_slug_list( $instance['subcategories'] ) : '';
			$hide_empty           = ! empty( $instance['hide_empty'] ) && 'yes' === $instance['hide_empty'];
			$desktop_image        = isset( $instance['desktop_image'] ) ? $instance['desktop_image'] : '';
			$mobile_image         = isset( $instance['mobile_image'] ) ? $instance['mobile_image'] : '';
			$mobile_grid_columns  = max( 1, min( 2, abs( $instance['mobile_grid_columns'] ?? '1' ) ) );
			$autoplay             = ( $instance['autoplay'] ?? 'no' ) === 'yes';
			$autoplay_hover_pause = ( $instance['autoplay_hover_pause'] ?? 'yes' ) === 'yes';
			$autoplay_timeout     = absint( $instance['autoplay_timeout'] ?? '5000' );
			$groups               = isset( $instance['groups'] ) && is_array( $instance['groups'] ) ? array_values( $instance['groups'] ) : array();

			$category = get_term_by( 'slug', sanitize_title( $category ), 'product_cat' );
			$children = array();

			if ( ! empty( $subcategories ) ) {
				$children = get_categories(
					array(
						'taxonomy'   => 'product_cat',
						'hide_empty' => $hide_empty,
						'slug'       => $subcategories,
						'number'     => 7,
					)
				);
			} elseif ( ! empty( $category ) ) {
				$children = get_categories(
					array(
						'taxonomy'   => 'product_cat',
						'hide_empty' => $hide_empty,
						'parent'     => absint( $category->term_id ),
						'number'     => 7,
					)
				);
			}

			$current_group = 0;

			if ( isset( $instance['current_group'] ) && ( $this->is_preview() || $this->is_elementor_editor() || wp_doing_ajax() ) ) {
				$current_group = absint( $instance['current_group'] );
				$current_group = isset( $groups[ $current_group ] ) ? $current_group : 0;
			}

			$header_groups = array();

			foreach ( $groups as $index => $header_group ) {
				$header_group = is_array( $header_group ) ? $header_group : array();

				$header_groups[] = array(
					'name'        => isset( $header_group['title'] ) ? $header_group['title'] : '',
					'active'      => $current_group === $index,
					'group_order' => isset( $header_group['group_order'] ) ? $header_group['group_order'] : 0,
					'data'        => array(
						'widget_id'   => $this->number,
						'group_index' => $index,
					),
				);
			}

			uasort(
				$header_groups,
				function( $a, $b ) {
					$a = intval( $a['group_order'] );
					$b = intval( $b['group_order'] );

					if ( $a === $b ) {
						return 0;
					}

					return ( $a < $b ) ? -1 : 1;
				}
			);

			$number     = 0;
			$show       = 'all';
			$products   = '';
			$categories = '';
			$orderby    = 'date';
			$order      = 'desc';

			if ( count( $groups ) && isset( $groups[ $current_group ] ) ) {
				$group      = $groups[ $current_group ];
				$number     = isset( $group['number'] ) ? absint( $group['number'] ) : $number;
				$show       = isset( $group['show'] ) ? sanitize_title( $group['show'] ) : $show;
				$products   = isset( $group['products'] ) ? wp_parse_slug_list( $group['products'] ) : $products;
				$categories = isset( $group['categories'] ) ? wp_parse_slug_list( $group['categories'] ) : $categories;
				$orderby    = isset( $group['orderby'] ) ? sanitize_title( $group['orderby'] ) : $orderby;
				$order      = isset( $group['order'] ) ? sanitize_title( $group['order'] ) : $order;
			}

			$product_visibility_term_ids = wc_get_product_visibility_term_ids();

			$query_args = array(
				'posts_per_page' => $number,
				'post_status'    => 'publish',
				'post_type'      => 'product',
				'no_found_rows'  => 1,
				'order'          => $order,
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
				'tax_query'      => array(
					'relation' => 'AND',
				),
			);

			if ( 'yes' === get_option( 'woocommerce_hide_out_of_stock_items' ) ) {
				$query_args['tax_query'][] = array(
					array(
						'taxonomy' => 'product_visibility',
						'field'    => 'term_taxonomy_id',
						'terms'    => $product_visibility_term_ids['outofstock'],
						'operator' => 'NOT IN',
					),
				);
			}

			if ( ! empty( $products ) ) {
				$query_args['post_name__in'] = $products;
			}

			if ( ! empty( $categories ) ) {
				$query_args['tax_query'][] = array(
					'taxonomy' => 'product_cat',
					'field'    => 'slug',
					'terms'    => $categories,
				);
			}

			switch ( $show ) {
				case 'featured':
					$query_args['tax_query'][] = array(
						'taxonomy' => 'product_visibility',
						'field'    => 'term_taxonomy_id',
						'terms'    => $product_visibility_term_ids['featured'],
					);
					break;
				case 'onsale':
					$product_ids_on_sale    = wc_get_product_ids_on_sale();
					$product_ids_on_sale[]  = 0;
					$query_args['post__in'] = $product_ids_on_sale;
					break;
			}

			switch ( $orderby ) {
				case 'price':
					// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$query_args['meta_key'] = '_price';
					$query_args['orderby']  = 'meta_value_num';
					break;
				case 'sales':
					// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$query_args['meta_key'] = 'total_sales';
					$query_args['orderby']  = 'meta_value_num';
					break;
				case 'rating':
					// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
					$query_args['meta_key'] = '_wc_average_rating';
					$query_args['orderby']  = 'meta_value_num';
					break;
				case 'product_slugs':
					$query_args['orderby'] = 'post_name__in';
					break;
				default:
					$query_args['orderby'] = 'date';
			}

			$products = new WP_Query( $query_args );

			$nonce = wp_create_nonce( 'redparts_sputnik_widget' );

			?>
			<!-- .th-block-zone -->
			<div
				class="th-block th-block-zone"
				data-ajax-url="<?php echo esc_url( apply_filters( 'redparts_sputnik_ajax_url', '' ) ); ?>"
				data-nonce="<?php echo esc_attr( $nonce ); ?>"
				data-autoplay="<?php echo esc_attr( $autoplay ? 1 : 0 ); ?>"
				data-autoplay-hover-pause="<?php echo esc_attr( $autoplay_hover_pause ? 1 : 0 ); ?>"
				data-autoplay-timeout="<?php echo esc_attr( $autoplay_timeout ); ?>"
				data-mobile-grid-columns="<?php echo esc_attr( $mobile_grid_columns ); ?>"
			>
				<div class="th-container">
					<div class="th-block-zone__body">
						<div class="th-block-zone__card th-category-card th-category-card--layout--overlay">
							<div class="th-category-card__body">
								<?php
								$desktop_image_url = wp_get_attachment_image_url( $desktop_image, 'full' );
								$mobile_image_url  = wp_get_attachment_image_url( $mobile_image, 'full' );

								$srcset = $mobile_image_url . ' 530w, ' . $desktop_image_url . ' 305w';
								?>
								<?php if ( ! empty( $desktop_image_url ) && ! empty( $mobile_image_url ) ) : ?>
									<div class="th-category-card__overlay-image">
										<img
											srcset="<?php echo esc_attr( $srcset ); ?>"
											src="<?php echo esc_attr( $desktop_image_url ); ?>"
											sizes="(max-width: 575px) 530px, 305px"
											loading="lazy"
											alt=""
											role="presentation"
										>
									</div>
									<div class="th-category-card__overlay-image th-category-card__overlay-image--blur">
										<img
											srcset="<?php echo esc_attr( $srcset ); ?>"
											src="<?php echo esc_attr( $desktop_image_url ); ?>"
											sizes="(max-width: 575px) 530px, 305px"
											loading="lazy"
											alt=""
											role="presentation"
										>
									</div>
								<?php endif; ?>

								<?php if ( $category ) : ?>
									<div class="th-category-card__content">
										<div class="th-category-card__info">
											<div class="th-category-card__name">
												<a href="<?php echo esc_url( get_term_link( $category, 'product_cat' ) ); ?>">
													<?php echo esc_html( $category->name ); ?>
												</a>
											</div>
											<?php if ( ! empty( $children ) ) : ?>
												<ul class="th-category-card__children">
													<?php foreach ( $children as $child ) : ?>
														<li>
															<a href="<?php echo esc_url( get_term_link( $child->term_id, 'product_cat' ) ); ?>">
																<?php echo esc_html( $child->name ); ?>
															</a>
														</li>
													<?php endforeach; ?>
												</ul>
											<?php endif; ?>
											<div class="th-category-card__actions">
												<a
													href="<?php echo esc_url( get_term_link( $category, 'product_cat' ) ); ?>"
													class="th-button th-button--style--primary th-button--size--small"
												>
													<?php echo esc_html__( 'Shop All', 'redparts-sputnik' ); ?>
												</a>
											</div>
										</div>
									</div>
								<?php endif; ?>
							</div>
						</div>
						<div class="th-block-zone__widget">
							<div class="th-block-zone__widget-header">
								<?php if ( ! empty( $header_groups ) ) : ?>
									<div class="th-block-zone__tabs">
										<?php foreach ( $header_groups as $header_group ) : ?>
											<?php
											$button_classes = array( 'th-block-zone__tabs-button' );

											if ( $header_group['active'] ) {
												$button_classes[] = 'th-block-zone__tabs-button--active';
											}
											?>
											<button
												type="button"
												class="<?php echo esc_attr( implode( ' ', $button_classes ) ); ?>"
												data-group-data="<?php echo esc_attr( wp_json_encode( $header_group['data'] ) ); ?>"
											>
												<?php echo esc_html( $header_group['name'] ); ?>
											</button>
										<?php endforeach; ?>
									</div>
								<?php endif; ?>

								<div class="th-arrow th-arrow--direction--prev th-block-zone__arrow th-block-zone__arrow--prev">
									<button class="th-arrow__button" type="button">
										<svg xmlns="http://www.w3.org/2000/svg" width="7" height="11">
											<path
												d="M6.7.3c-.4-.4-.9-.4-1.3 0L0 5.5l5.4 5.2c.4.4.9.3 1.3 0 .4-.4.4-1 0-1.3l-4-3.9 4-3.9c.4-.4.4-1 0-1.3z"></path>
										</svg>
									</button>
								</div>
								<div class="th-arrow th-arrow--direction--next th-block-zone__arrow th-block-zone__arrow--next">
									<button class="th-arrow__button" type="button">
										<svg xmlns="http://www.w3.org/2000/svg" width="7" height="11">
											<path
												d="M.3 10.7c.4.4.9.4 1.3 0L7 5.5 1.6.3C1.2-.1.7 0 .3.3c-.4.4-.4 1 0 1.3l4 3.9-4 3.9c-.4.4-.4 1 0 1.3z"></path>
										</svg>
									</button>
								</div>
							</div>
							<div class="th-block-zone__widget-body">
								<div class="th-block-zone__carousel">
									<div class="th-block-zone__carousel-loader"></div>
									<div class="owl-carousel">
										<?php while ( $products->have_posts() ) : ?>
											<ul class="th-block-zone__carousel-item">
												<?php
												$products->the_post();

												$exclude_elements = array(
													'featured_attributes',
													'description',
													'footer_wishlist',
													'footer_compare',
												);

												wc_set_loop_prop( 'redparts_product_card_exclude', $exclude_elements );

												wc_get_template_part( 'content', 'product' );
												?>
											</ul>
										<?php endwhile; ?>
									</div>
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>
			<!-- .th-block-zone / end -->
			<?php

			wp_reset_postdata();
			wc_reset_loop();
		}

		/**
		 * Returns form fields.
		 *
		 * @noinspection DuplicatedCode
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'   => 'category',
					'label' => esc_html__( 'Category', 'redparts-sputnik' ),
					'type'  => 'text',
				),
				array(
					'key'   => 'subcategories',
					'label' => esc_html__( 'Subcategories (comma separated slugs)', 'redparts-sputnik' ),
					'type'  => 'textarea',
				),
				array(
					'key'     => 'hide_empty',
					'label'   => esc_html__( 'Hide empty categories', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
				),
				array(
					'key'   => 'desktop_image',
					'label' => esc_html__( 'Desktop image', 'redparts-sputnik' ),
					'type'  => 'image',
				),
				array(
					'key'   => 'mobile_image',
					'label' => esc_html__( 'Mobile image', 'redparts-sputnik' ),
					'type'  => 'image',
				),
				array(
					'key'     => 'mobile_grid_columns',
					'label'   => esc_html__( 'Mobile grid columns', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => '1',
							'label' => esc_html__( '1 column', 'redparts-sputnik' ),
						),
						array(
							'key'   => '2',
							'label' => esc_html__( '2 columns', 'redparts-sputnik' ),
						),
					),
					'default' => '1',
				),
				array(
					'key'     => 'autoplay',
					'label'   => esc_html__( 'Autoplay', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'no',
				),
				array(
					'key'     => 'autoplay_hover_pause',
					'label'   => esc_html__( 'Pause autoplay on hover', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'yes',
				),
				array(
					'key'     => 'autoplay_timeout',
					'label'   => esc_html__( 'Autoplay timeout', 'redparts-sputnik' ),
					'type'    => 'number',
					'default' => '5000',
				),
				array(
					'key'        => 'groups',
					'select_key' => 'current_group',
					'label'      => esc_html__( 'Products groups', 'redparts-sputnik' ),
					'type'       => 'tabs',
					'fields'     => array(
						array(
							'key'          => 'title',
							'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'     => 'group_order',
							'label'   => esc_html__( 'Group order', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
						array(
							'key'     => 'number',
							'label'   => esc_html__( 'Number of products to show', 'redparts-sputnik' ),
							'type'    => 'number',
							'default' => 0,
						),
						array(
							'key'     => 'show',
							'label'   => esc_html__( 'Show', 'redparts-sputnik' ),
							'type'    => 'select',
							'default' => 'all',
							'options' => array(
								array(
									'key'   => 'all',
									'label' => esc_html__( 'All products', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'featured',
									'label' => esc_html__( 'Featured products', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'onsale',
									'label' => esc_html__( 'On-sale products', 'redparts-sputnik' ),
								),
							),
						),
						array(
							'key'   => 'products',
							'label' => esc_html__( 'Products (comma separated slugs)', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'   => 'categories',
							'label' => esc_html__( 'Categories (comma separated slugs)', 'redparts-sputnik' ),
							'type'  => 'text',
						),
						array(
							'key'     => 'orderby',
							'label'   => esc_html__( 'Order by', 'redparts-sputnik' ),
							'type'    => 'select',
							'default' => 'date',
							'options' => array(
								array(
									'key'   => 'date',
									'label' => esc_html__( 'Date', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'price',
									'label' => esc_html__( 'Price', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'sales',
									'label' => esc_html__( 'Sales', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'rating',
									'label' => esc_html__( 'Average rating', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'product_slugs',
									'label' => esc_html__( 'Product slugs', 'redparts-sputnik' ),
								),
							),
						),
						array(
							'key'     => 'order',
							'label'   => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'    => 'select',
							'default' => 'desc',
							'options' => array(
								array(
									'key'   => 'desc',
									'label' => esc_html__( 'DESC', 'redparts-sputnik' ),
								),
								array(
									'key'   => 'asc',
									'label' => esc_html__( 'ASC', 'redparts-sputnik' ),
								),
							),
						),
					),
					'l10n'       => array(
						'add'    => esc_html__( 'Add Group', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Group', 'redparts-sputnik' ),
						/* translators: %s: Group number. */
						'title'  => esc_html__( 'Group %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Enqueue admin scripts.
		 */
		public function enqueue_admin_scripts() {
			wp_enqueue_style( 'redparts-sputnik-admin' );
			wp_enqueue_script( 'redparts-sputnik-admin' );
		}

		/**
		 * Enqueue frontend scripts.
		 */
		public function enqueue_frontend_scripts() {
			wp_enqueue_script( 'redparts-sputnik-widget-block-zone' );
		}
	}
}
