<?php
/**
 * Main WPML integration file.
 *
 * @package RedParts\Sputnik
 * @since 1.6.0
 */

namespace RedParts\Sputnik\WPML;

use RedParts\Sputnik\Singleton;
use RedParts\Sputnik\Plugin;
use WP_Term;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\WPML\WPML' ) ) {
	/**
	 * Class WPML
	 *
	 * @since 1.6.0
	 */
	class WPML extends Singleton {
		/**
		 * Initialization.
		 *
		 * @since 1.6.0
		 */
		public function init() {
			add_filter( 'wpml_elementor_widgets_to_translate', array( $this, 'wpml_elementor_widgets_to_translate' ) );
			add_filter( 'redparts_sputnik_ajax_url', array( $this, 'ajax_url' ) );
		}

		/**
		 * Modifies an array with information needed to translate Elementor widgets.
		 *
		 * @since 1.6.0
		 *
		 * @param array $widgets An array with information needed to translate Elementor widgets.
		 * @return mixed
		 */
		public function wpml_elementor_widgets_to_translate( array $widgets ): array {
			foreach ( Plugin::instance()->get_widget_classes() as $widget_class ) {
				$widget      = new $widget_class();
				$widget_type = 'wp-widget-' . $widget->id_base;

				$widgets[ $widget_type ] = array(
					'conditions'        => array( 'widgetType' => $widget_type ),
					'fields'            => array(),
					'integration-class' => '\RedParts\Sputnik\WPML\Widgets_Module',
				);
			}

			return $widgets;
		}

		/**
		 * Adds lang query parameter to the AJAX url.
		 *
		 * @since 1.6.0
		 *
		 * @param string $url AJAX url.
		 * @return string
		 */
		public function ajax_url( string $url = '' ): string {
			if ( empty( $url ) ) {
				$url = admin_url( 'admin-ajax.php' );
			}

			$current_language_code = apply_filters( 'wpml_current_language', null );

			if ( $current_language_code ) {
				$url = add_query_arg( 'redparts_sputnik_lang', $current_language_code, $url );
			}

			return $url;
		}

		/**
		 * Switches the language for AJAX requests.
		 *
		 * @since 1.6.0
		 */
		public static function switch_ajax_language() {
			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			if ( isset( $_GET['redparts_sputnik_lang'] ) ) {
				do_action( 'wpml_switch_language', sanitize_key( wp_unslash( $_GET['redparts_sputnik_lang'] ) ) );
			}
			// phpcs:enable
		}

		/**
		 * Returns the original product ID based on the passed original or translated product ID.
		 *
		 * @since 1.6.0
		 *
		 * @param int $product_id Original or translated product ID.
		 * @return int
		 */
		public static function get_original_product_id( int $product_id ): int {
			$default_language_code = apply_filters( 'wpml_default_language', null );

			return absint( apply_filters( 'wpml_object_id', $product_id, 'product', true, $default_language_code ) );
		}

		/**
		 * Returns the original term ID based on the passed original or translated term ID.
		 *
		 * @since 1.6.0
		 *
		 * @param int|WP_Term $term Original or translated term ID or WP_Term object.
		 * @return int
		 */
		public static function get_original_term_id( $term ): int {
			$term                  = get_term( $term );
			$default_language_code = apply_filters( 'wpml_default_language', null );

			return absint( apply_filters( 'wpml_object_id', $term->term_id, $term->taxonomy, true, $default_language_code ) );
		}
	}
}
