<?php

if ( !defined( 'ABSPATH' ) ) {
	exit;
}

add_action( 'wp_ajax_send_email_async', 'auroship_send_email_async' );
add_action( 'wp_ajax_nopriv_send_email_async', 'auroship_send_email_async' );

/**
 * Email ajax call response
 *
 * @return void
 */
function auroship_send_email_async() {
	$order_id = isset( $_POST['order_id'] ) ? intval( $_POST['order_id'] ) : 0;

	if ( $order_id ) {
		// Sanitize and validate the email data
		$email_data = isset( $_POST['email_data'] ) ? wp_unslash( $_POST['email_data'] ) : array();

		if ( is_array( $email_data ) ) {
			foreach ( $email_data as $key => $value ) {
				$email_data[$key] = sanitize_text_field( $value );
			}
			// Send the email
			auroship_send_multiple_emails( $email_data );

			wp_send_json_success( array( 'message' => 'Email sent successfully.' ) );
		} else {
			wp_send_json_error( array( 'message' => 'Invalid email data format.' ) );
		}

	} else {
		wp_send_json_error( array( 'message' => 'Missing data to send email.' ) );
	}

	wp_die();
}

/**
 * Email template
 *
 * @param string $recipient_name Email receiver name.
 * @param string $order_number Unique order number.
 * @param string $tracking_number Tracking number.
 * @param string $delivery_date Delivery date.
 * @param string $order_url Order URL.
 * @param string $recipient Mail receiver.
 * @param string $order_status Order status.
 *
 * @return string HTML email template.
 */
function get_email_template( $recipient_name, $order_number, $tracking_number, $delivery_date, $order_url, $recipient, $order_status ) {
	$site_title = get_bloginfo( 'name' );
	$woocommerce_text = get_option( 'woocommerce_email_footer_text' );
	$output_text = str_replace( '{site_title}', esc_html( $site_title ), esc_html( $woocommerce_text ) );

	$msg = ( 'buyer' === $recipient ) ? __( 'Your order', 'auroship' ) : __( 'Order', 'auroship' );

	$delivery_schedule = '';
	if ( 'delivered' !== strtolower( $order_status ) ) {
		$delivery_schedule = '<p>' . __( 'Estimated Delivery Schedule: ', 'auroship' ) . esc_html( $delivery_date ) . '</p>';
	}

	return '<html lang="en">
    <body>
        <div class="main-email-div" style="background-color: #f7f7f7; padding: 0; margin: 0; width: 100%;">
            <div style="max-width: 600px; margin: 0 auto; padding: 0;">
                <table style="background-color: #ffffff; border: 1px solid #dedede; border-radius: 3px; box-shadow: 0 1px 4px rgba(0, 0, 0, 0.1); width: 100%;">
                    <!-- Header -->
                    <tr>
                        <td style="padding: 36px 48px; color: #ffffff; border-radius: 3px 3px 0 0; background-color: #7f54b3;">
                            <h1 style="margin: 0; font-size: 30px; font-weight: 300;">' . sprintf( __( 'Hi %s,', 'auroship' ), esc_html( $recipient_name ) ) . '</h1>
                        </td>
                    </tr>
                    <!-- Body -->
                    <tr>
                        <td style="padding: 36px; color: #636363; font-family: Arial, sans-serif; font-size: 14px; line-height: 150%;">
                            <p>' . esc_html( $msg ) . ' #' . esc_html( $order_number ) . ' is ' . esc_html( $order_status ) . '.</p>
                            <p>' . __( 'AWB No: ', 'auroship' ) . esc_html( $tracking_number ) . '</p>
                            ' . $delivery_schedule . '
                            <p>' . __( 'Track ', 'auroship' ) . strtolower( esc_html( $msg ) ) . ' ' . __( 'at ', 'auroship' ) . '<a href="' . esc_url( $order_url ) . '">' . esc_html( $site_title ) . '</a></p></br></br>
                            <p style="margin: 0; padding: 0; font-size: 14px; line-height: 150%;">' . __( 'Regards,', 'auroship' ) . '</p>
                            <p style="margin: 0; padding: 0; font-size: 14px; line-height: 150%;">' . esc_html( $site_title ) . '</p>
                        </td>
                    </tr>
                    <!-- Footer -->
                    <tr>
                        <td style="padding: 5px; text-align: center; color: #8a8a8a; font-size: 12px; line-height: 150%; border-radius: 0 0 3px 3px;">
                            <p>' . esc_html( $output_text ) . '</p>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
    </body>
</html>';
}

/**
 * Send emails to buyer and seller
 *
 *  @param array $email_data Email data.
 *
 * @return void
 */
function auroship_send_multiple_emails( $email_data ) {
	$site_title = get_bloginfo( 'name' );

	if ( in_array( 'dokan-lite/dokan.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) && function_exists( 'dokan' ) ) {
		$base_url = dokan_get_navigation_url( 'orders' );

		$view_mode = 'email';
		$permission = 1;

		$order_url = add_query_arg(
			array(
				'order_id' => $email_data['order_number'],
				'_view_mode' => $view_mode,
				'permission' => $permission,
			),
			$base_url
		);

	} else {
		$order_url = add_query_arg(
			[
				'post' => $email_data['order_number'],
				'action' => 'edit',
			],
			admin_url( 'post.php' )
		);
	}

	$recipients = array(
		array(
			'email' => sanitize_email( $email_data['buyer_email'] ),
			'subject' => 'Order ' . $email_data['order_number'] . ' confirmed for ' . $site_title,
			'message' => get_email_template(
				$email_data['buyer_name'],
				$email_data['order_number'],
				$email_data['tracking_number'],
				$email_data['delivery_date'],
				wc_get_account_endpoint_url( 'view-order' ) . esc_html( $email_data['order_number'] ),
				'buyer',
				$email_data['order_status']

			),
		),
		array(
			'email' => sanitize_email( $email_data['seller_email'] ),
			'subject' => 'Tracking number for order #' . $email_data['order_number'],
			'message' => get_email_template(
				$email_data['seller_name'],
				$email_data['order_number'],
				$email_data['tracking_number'],
				$email_data['delivery_date'],
				$order_url,
				'seller',
				$email_data['order_status']
			),
			'attachments' => 'yes',
		),
	);

	auroship_send_emails_to_multiple_recipients( $recipients, $email_data['order_number'] );
}

/**
 * Send emails to multiple recipients
 *
 * @param array $recipients Recipients data.
 *
 * @return void
 */
function auroship_send_emails_to_multiple_recipients( $recipients, $order_id ) {
	$headers = array( 'Content-Type: text/html; charset=UTF-8;' );

	foreach ( $recipients as $recipient ) {
		$to = $recipient['email'];
		$subject = $recipient['subject'];
		$message = $recipient['message'];
		$pdf_url = get_post_meta( $order_id, '_pdf_file_url', true );
		$attachments = array();
		$upload_dir = wp_upload_dir();
		$pdf_file_path = str_replace( $upload_dir['baseurl'], $upload_dir['basedir'], $pdf_url );

		if ( isset( $recipient['attachments'] ) && !empty( $pdf_url ) ) {
			$attachments = array( $pdf_file_path );
		}
		if ( !wp_mail( $to, $subject, $message, $headers, $attachments ) ) {
			Auroship_utilities::log( 'email not sent' );
		}
	}
}
