<?php

/**
 * An external standard for AuroShip.
 *
 * @package AuroShip
 * @link    https://www.aurobees.com/
 */

if ( !defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Communicates with amazon shipping API
 *
 * @package AuroShip
 * @link    https://www.aurobees.com/
 */
class Auroship_API {
	/**
	 * Get access token
	 *
	 * @param string $auth_token authentication token for amazon api
	 * @param string $apiUrl      api url
	 *
	 * @return array
	 */
	public static function authenticate_amazon_api() {
		$api_url = get_option( 'auroship_api_url' );
		$params = [
			'client_id' => get_option( 'auroship_client_id' ),
			'client_secret' => get_option( 'auroship_client_secret' ),
			'refresh_token' => get_option( 'auroship_refresh_token' ),
			'grant_type' => 'refresh_token',
		];
		$api = '/auth/o2/token/';
		$endpoint = $api_url . $api;

		try {
			$response = wp_remote_post(
				$endpoint,
				[
					'method' => 'POST',
					'headers' => array(
						'Content-Type' => 'application/x-www-form-urlencoded',
					),
					'body' => $params,
					'timeout' => 50,
				]
			);

			if ( 200 != $response['response']['code'] && 201 != $response['response']['code'] && 204 != $response['response']['code'] ) {
				Auroship_utilities::log( 'api:' . $api . '#ERROR:' . $response['body'] );
			} else {
				$response = json_decode( $response['body'], true );
				$expiration_timestamp = time() + $response['expires_in'];
				$token_data = array(
					'auroship_amzn_auth_token' => $response['access_token'],
					'expires_at' => $expiration_timestamp,
				);

				// Save the array to the options table
				update_option( 'auroship_token_data', $token_data );
				return $response;
			}
		} catch ( Exception $e ) {
			Auroship_utilities::log( 'called api auth:' . $api . '#response:error' );
			return array( 'error' => true );
		}
	}
	/**
	 * Get access token if not expired
	 *
	 * @param string $auth_token authentication token for amazon api
	 *
	 * @return string
	 */
	public static function get_generated_token() {
		$token_data = get_option( 'auroship_token_data' );

		// Check if the data exists and is valid
		if ( $token_data && isset( $token_data['auroship_amzn_auth_token'], $token_data['expires_at'] ) ) {
			// Check if the token is still valid
			if ( time() < $token_data['expires_at'] ) {
				$auth_token = $token_data['auroship_amzn_auth_token'];
			} else {
				// Token has expired then generate new token
				$token = self::authenticate_amazon_api();
				if ( empty( $token['error'] ) && !empty( $token['access_token'] ) ) {
					$auth_token = $token['access_token'];
				} else {
					die( esc_html( __( 'An error occurred while retrieving the access token.', 'auroship' ) ) );
				}
			}
		} else {
			$token = self::authenticate_amazon_api();
			if ( empty( $token['error'] ) && !empty( $token['access_token'] ) ) {
				$auth_token = $token['access_token'];
			} else {
				die( esc_html( __( 'An error occurred while retrieving the access token.', 'auroship' ) ) );
			}
		}
		return $auth_token;
	}

	/**
	 * Get the package label
	 *
	 * @param string $auth_token authentication token for amazon api
	 * @param string $apiUrl      api url
	 *
	 * @return array
	 */
	public static function get_shipping_label( $params, $order_id ) {
		$api_url = get_option( 'auroship_seller_api_url' );
		$api = '/shipping/v2/oneClickShipment/';
		Auroship_utilities::log( 'order_id' . ':' . $order_id . '_calling_api ' . $api );
		$auth_token = self::get_generated_token();
		$amazon_shipping_business_id = get_option( 'auroship_amzn_shipng_bsns_id' );
		$headers = array( 'x-amz-access-token' => $auth_token, 'x-amzn-shipping-business-id' => $amazon_shipping_business_id, 'content-type' => 'application/json' );
		$endpoint = $api_url . $api;

		try {
			$response = wp_remote_post(
				$endpoint,
				[
					'method' => 'POST',
					'headers' => $headers,
					'body' => json_encode( $params ),
					'timeout' => 50,
				]
			);

			if ( 200 != $response['response']['code'] && 201 != $response['response']['code'] && 204 != $response['response']['code'] ) {
				Auroship_utilities::log( 'api:' . $api . '##ERROR:' . $response['body'] . '###Params:' . json_encode( $params ) );
				$response = json_decode( $response['body'] );
			} else {
				$response = json_decode( $response['body'] );
				update_post_meta( $order_id, '_delivery_start_date', $response->payload->promise->deliveryWindow->start );
				update_post_meta( $order_id, '_delivery_end_date', $response->payload->promise->deliveryWindow->end );
				update_post_meta( $order_id, '_carrier_id', $response->payload->carrier->id );
				update_post_meta( $order_id, '_shipping_provider', $response->payload->carrier->name );
				update_post_meta( $order_id, '_tracking_id', $response->payload->packageDocumentDetails[0]->trackingId );
				update_post_meta( $order_id, '_bas64_string', $response->payload->packageDocumentDetails[0]->packageDocuments[0]->contents );
				Auroship_utilities::log( 'order_id' . ':' . $order_id . '_called_api ' . $api );
				return array( 'success' => true, 'message' => 'successfully fetch the shipping label' );
			}
		} catch ( Exception $e ) {
			Auroship_utilities::log( 'called api oneclick:' . $api . '##response:error' );
			return array( 'error' => true, 'message' => $response->errors[0]->message, 'detail_error_message' => $response->errors[0]->details );
		}
	}

	/**
	 * Get the tracking info
	 *
	 * @param string $auth_token authentication token for amazon api
	 * @param string $apiUrl api url
	 * @param string $params params required to fetch tracking info
	 * @param int $order_id unique id
	 *
	 * @return array
	 */
	public static function get_shipping_tracking( $params, $order_id ) {
		$api_url = get_option( 'auroship_seller_api_url' );
		$api = '/shipping/v2/tracking/';
		Auroship_utilities::log( 'order_id' . ':' . $order_id . '_calling_api ' . $api );
		$auth_token = self::get_generated_token();
		$amazon_shipping_business_id = get_option( 'auroship_amzn_shipng_bsns_id' );
		$headers = array( 'x-amz-access-token' => $auth_token, 'x-amzn-shipping-business-id' => $amazon_shipping_business_id, 'content-type' => 'application/json' );
		$endpoint = add_query_arg( $params, $api_url . $api );

		try {
			$response = wp_remote_get(
				$endpoint,
				array(
					'headers' => $headers,
					'timeout' => 50,
				)
			);

			if ( 200 != $response['response']['code'] && 201 != $response['response']['code'] && 204 != $response['response']['code'] ) {
				Auroship_utilities::log( 'api:' . $api . '###ERROR:' . $response['body'] . '###Params:' . json_encode( $params ) );
				$response = json_decode( $response['body'] );
			} else {
				update_post_meta( $order_id, '_tracking_history', $response['body'] );
				$response = json_decode( $response['body'] );

				update_post_meta( $order_id, '_promised_delivery_date', $response->payload->promisedDeliveryDate );
				update_post_meta( $order_id, '_shipping_status', $response->payload->summary->status );
				Auroship_utilities::log( 'order_id' . ':' . $order_id . '_called_api ' . $api );
				return array( 'success' => true, 'message' => 'successfully fetch the tracking info' );
			}
		} catch ( Exception $e ) {
			Auroship_utilities::log( 'called api tracking:' . $api . '###response:error' );
			return array( 'error' => true, 'message' => $response->errors[0]->message, 'detail_error_message' => $response->errors[0]->details );
		}
	}
}
