<?php
/**
 * This file contains code related to product comparison.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;
use WC_Product;
use WC_Product_Attribute;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Compare' ) ) {
	/**
	 * Class Compare
	 */
	class Compare extends Singleton {
		const COOKIE_KEY = 'redparts_sputnik_compare';

		/**
		 * List of product ids for comparison.
		 *
		 * @var array
		 */
		protected $products_list = array();

		/**
		 * Initialization.
		 */
		public function init() {
			add_action( 'init', array( $this, 'deferred_init' ) );
		}

		/**
		 * Deferred initialization.
		 *
		 * @since 1.5.0
		 */
		public function deferred_init() {
			if ( ! $this->is_enabled() ) {
				return;
			}

			add_action( 'redparts_sputnik_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

			add_shortcode( 'redparts_sputnik_compare', array( $this, 'shortcode' ) );

			if ( wp_doing_ajax() ) {
				add_action( 'wp_ajax_redparts_sputnik_compare_add', array( $this, 'ajax' ) );
				add_action( 'wp_ajax_nopriv_redparts_sputnik_compare_add', array( $this, 'ajax' ) );

				add_action( 'wp_ajax_redparts_sputnik_compare_remove', array( $this, 'ajax' ) );
				add_action( 'wp_ajax_nopriv_redparts_sputnik_compare_remove', array( $this, 'ajax' ) );

				add_action( 'wp_ajax_redparts_sputnik_compare_clear', array( $this, 'ajax' ) );
				add_action( 'wp_ajax_nopriv_redparts_sputnik_compare_clear', array( $this, 'ajax' ) );
			}

			// Button content.
			add_action( 'redparts_sputnik_compare_button_content', array( $this, 'the_button_label' ), 100 );
			add_action( 'redparts_sputnik_compare_button_content', array( $this, 'the_button_icon' ), 200 );

			// Initializes the product list.
			if ( isset( $_COOKIE[ self::COOKIE_KEY ] ) ) {
				$this->products_list = wp_parse_id_list( wp_unslash( $_COOKIE[ self::COOKIE_KEY ] ) );
			}
		}

		/**
		 * Returns true if compare module enabled.
		 *
		 * @since 1.5.0
		 *
		 * @return bool
		 */
		public function is_enabled(): bool {
			return 'no' !== Settings::instance()->get( 'compare_enabled', 'yes' );
		}

		/**
		 * Enqueue scripts.
		 */
		public function enqueue_scripts() {
			wp_register_script(
				'redparts-sputnik-compare',
				Plugin::instance()->url( 'assets/js/compare.js' ),
				array( 'jquery' ),
				Plugin::VERSION,
				true
			);
			wp_localize_script(
				'redparts-sputnik-compare',
				'redPartsSputnikCompareVars',
				array(
					'ajaxUrl'     => apply_filters( 'redparts_sputnik_ajax_url', '' ),
					'pageUrl'     => $this->get_page_url(),
					'nonceAdd'    => wp_create_nonce( 'redparts_sputnik_compare_add' ),
					'nonceRemove' => wp_create_nonce( 'redparts_sputnik_compare_remove' ),
					'nonceClear'  => wp_create_nonce( 'redparts_sputnik_compare_clear' ),
					'button'      => array(
						'added' => array(
							'label' => apply_filters(
								'redparts_sputnik_compare_button_added_label',
								esc_html__( 'Go to compare', 'redparts-sputnik' )
							),
							'icon'  => apply_filters(
								'redparts_sputnik_compare_button_added_icon',
								''
							),
						),
					),
				)
			);

			wp_enqueue_script( 'redparts-sputnik-compare' );
		}

		/**
		 * Returns number of products in wish list.
		 *
		 * @return int
		 */
		public function count(): int {
			return count( $this->products_list );
		}

		/**
		 * Shortcode to display product comparison table.
		 */
		public function shortcode() {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return '';
			}

			global $product;

			$get_translated_product_id = function( $product_id ) {
				return apply_filters( 'wpml_object_id', $product_id, 'product', true );
			};

			/** Products array. @var WC_Product[] $products */
			$products = array_map( 'wc_get_product', array_map( $get_translated_product_id, $this->products_list ) );
			$products = array_filter(
				$products,
				function( $product ) {
					return ! ! $product;
				}
			);

			$rows                = array_map( 'trim', explode( ',', Settings::instance()->get( 'compare_rows', '' ) ) );
			$excluded_attributes = array_map( 'trim', explode( ',', Settings::instance()->get( 'compare_excluded_attributes', '' ) ) );

			$attributes_list   = array();
			$attributes_values = array();

			foreach ( $products as $product ) {
				/** Attributes array. @var WC_Product_Attribute[] $attributes */
				$attributes = $product->get_attributes();

				foreach ( $attributes as $attribute ) {
					if ( in_array( $attribute->get_taxonomy(), $excluded_attributes, true ) ) {
						continue;
					}

					$values = array();

					if ( ! array_key_exists( $attribute->get_id(), $attributes_list ) ) {
						$attributes_list[ $attribute->get_id() ] = wc_attribute_label( $attribute->get_name() );
					}

					if ( $attribute->is_taxonomy() ) {
						$attribute_values = wc_get_product_terms( $product->get_id(), $attribute->get_name(), array( 'fields' => 'all' ) );

						foreach ( $attribute_values as $attribute_value ) {
							$values[] = esc_html( $attribute_value->name );
						}
					} else {
						foreach ( $attribute->get_options() as $value ) {
							$values[] = esc_html( $value );
						}
					}

					if ( ! isset( $attributes_values[ $product->get_id() ] ) ) {
						$attributes_values[ $product->get_id() ] = array();
					}

					$attributes_values[ $product->get_id() ][ $attribute->get_id() ] = implode( ', ', $values );
				}
			}

			$display_row = function( $label, $get_value ) use ( $products ) {
				$is_identical = true;
				$values       = array();

				foreach ( $products as $product ) {
					$value = $get_value( $product );

					if ( ! empty( $values ) ) {
						$is_identical = $is_identical && in_array( $value, $values, true );
					}

					$values[] = $value;
				}

				$is_identical = $is_identical && 1 < count( $values );
				$row_classes  = array( 'th-compare-table__row' );

				if ( $is_identical ) {
					$row_classes[] = 'th-compare-table__row--identical';
				}

				?>
				<tr class="<?php redparts_sputnik_the_classes( ...$row_classes ); ?>">
					<th class="th-compare-table__column th-compare-table__column--header">
						<?php echo esc_html( $label ); ?>
					</th>
					<?php foreach ( $products as $product ) : ?>
						<?php $value = $get_value( $product ); ?>
						<td class="th-compare-table__column th-compare-table__column--product">
							<?php if ( ! empty( $value ) ) : ?>
								<?php echo esc_html( $value ); ?>
							<?php else : ?>
								<div class="th-compare-table__dash"></div>
							<?php endif; ?>
						</td>
					<?php endforeach; ?>
					<td class="th-compare-table__column th-compare-table__column--fake"></td>
				</tr>
				<?php
			};

			ob_start();

			$show_only_different = isset( $_COOKIE['th-compare-show'] ) && 'different' === $_COOKIE['th-compare-show'];

			?>
			<div class="th-block-compare">
				<?php if ( $products ) : ?>
					<div class="th-container">
						<div class="th-compare">
							<div class="th-compare__options-list">
								<div class="th-compare__option">
									<div class="th-compare__option-label">
										<?php esc_html_e( 'Show:', 'redparts-sputnik' ); ?>
									</div>
									<div class="th-compare__option-control">
										<div class="th-button-toggle">
											<div class="th-button-toggle__list">
												<label class="th-button-toggle__item">
													<input
														type="radio"
														class="th-button-toggle__input"
														name="th-compare-show"
														value="all"
														<?php checked( $show_only_different, false, true ); ?>
													>
													<span class="th-button-toggle__button">
														<?php esc_html_e( 'All', 'redparts-sputnik' ); ?>
													</span>
												</label>
												<label class="th-button-toggle__item">
													<input
														type="radio"
														class="th-button-toggle__input"
														name="th-compare-show"
														value="different"
														<?php checked( $show_only_different, true, true ); ?>
													>
													<span class="th-button-toggle__button">
														<?php esc_html_e( 'Different', 'redparts-sputnik' ); ?>
													</span>
												</label>
											</div>
										</div>
									</div>
								</div>
								<div class="th-compare__option">
									<div class="th-compare__option-control">
										<button
											type="button"
											class="th-button th-button--style--secondary th-button--size--extra-small th-compare-clear"
										>
											<?php esc_html_e( 'Clear list', 'redparts-sputnik' ); ?>
										</button>
									</div>
								</div>
							</div>

							<?php
							/**
							 * Hook: redparts_sputnik_compare_before_table.
							 *
							 * @hooked woocommerce_output_all_notices - 10
							 */
							do_action( 'redparts_sputnik_compare_before_table' );
							?>
							<div class="th-table-responsive">
								<?php

								$table_classes = array( 'th-compare__table', 'th-compare-table' );

								if ( $show_only_different ) {
									$table_classes[] = 'th-compare-table--only-different';
								}

								?>
								<table class="<?php redparts_sputnik_the_classes( ...$table_classes ); ?>">
									<tbody>
									<tr class="th-compare-table__row">
										<th class="th-compare-table__column th-compare-table__column--header">
											<?php esc_html_e( 'Product', 'redparts-sputnik' ); ?>
										</th>
										<?php foreach ( $products as $product ) : ?>
											<td class="th-compare-table__column th-compare-table__column--product">
												<a href="<?php echo esc_url( $product->get_permalink() ); ?>" class="th-compare-table__product">
													<div class="th-compare-table__product-image">
														<?php echo wp_kses( $product->get_image(), 'redparts_sputnik_image' ); ?>
													</div>
													<div class="th-compare-table__product-name">
														<?php echo esc_html( $product->get_title() ); ?>
													</div>
												</a>
											</td>
										<?php endforeach; ?>
										<td class="th-compare-table__column th-compare-table__column--fake"></td>
									</tr>

									<?php foreach ( $rows as $row_type ) : ?>
										<?php if ( 'rating' === $row_type ) : ?>
											<tr class="th-compare-table__row">
												<th class="th-compare-table__column th-compare-table__column--header">
													<?php esc_html_e( 'Rating', 'redparts-sputnik' ); ?>
												</th>
												<?php foreach ( $products as $product ) : ?>
													<?php $rating = $product->get_rating_count(); ?>
													<td class="th-compare-table__column th-compare-table__column--product">
														<?php if ( $product->get_rating_count() ) : ?>
															<div class="th-compare-table__rating">
																<div class="th-compare-table__rating-stars">
																	<?php wc_get_template( 'loop/rating.php' ); ?>
																</div>
																<div class="th-compare-table__rating-title">
																	<?php
																	echo esc_html(
																		sprintf(
																			// translators: %s: Number of reviews.
																			_n( '%s Review', '%s Reviews', $rating, 'redparts-sputnik' ), // SKIP-ESC.
																			$rating
																		)
																	);
																	?>
																</div>
															</div>
														<?php else : ?>
															<div class="th-compare-table__rating-title">
																<?php esc_html_e( 'No reviews', 'redparts-sputnik' ); ?>
															</div>
														<?php endif; ?>
													</td>
												<?php endforeach; ?>
												<td class="th-compare-table__column th-compare-table__column--fake"></td>
											</tr>
										<?php elseif ( 'availability' === $row_type ) : ?>
											<tr class="th-compare-table__row">
												<th class="th-compare-table__column th-compare-table__column--header">
													<?php esc_html_e( 'Availability', 'redparts-sputnik' ); ?>
												</th>
												<?php foreach ( $products as $product ) : ?>
													<td class="th-compare-table__column th-compare-table__column--product">
														<?php $stock_html = wc_get_stock_html( $product ); ?>
														<?php if ( $stock_html ) : ?>
															<?php echo wp_kses( $stock_html, 'redparts_sputnik_stock' ); ?>
														<?php else : ?>
															<div class="th-compare-table__dash"></div>
														<?php endif; ?>
													</td>
												<?php endforeach; ?>
												<td class="th-compare-table__column th-compare-table__column--fake"></td>
											</tr>
										<?php elseif ( 'price' === $row_type ) : ?>
											<tr class="th-compare-table__row">
												<th class="th-compare-table__column th-compare-table__column--header">
													<?php esc_html_e( 'Price', 'redparts-sputnik' ); ?>
												</th>
												<?php foreach ( $products as $product ) : ?>
													<td class="th-compare-table__column th-compare-table__column--product">
														<?php woocommerce_template_loop_price(); ?>
													</td>
												<?php endforeach; ?>
												<td class="th-compare-table__column th-compare-table__column--fake"></td>
											</tr>
										<?php elseif ( 'add_to_cart' === $row_type ) : ?>
											<tr class="th-compare-table__row">
												<th class="th-compare-table__column th-compare-table__column--header">
													<?php esc_html_e( 'Add To Cart', 'redparts-sputnik' ); ?>
												</th>
												<?php foreach ( $products as $product ) : ?>
													<td class="th-compare-table__column th-compare-table__column--product">
														<?php woocommerce_template_loop_add_to_cart(); ?>
													</td>
												<?php endforeach; ?>
												<td class="th-compare-table__column th-compare-table__column--fake"></td>
											</tr>
										<?php elseif ( 'sku' === $row_type ) : ?>
											<?php
											$display_row(
												esc_html__( 'SKU', 'redparts-sputnik' ),
												function( $product ) {
													return $product->get_sku();
												}
											);
											?>
										<?php elseif ( 'weight' === $row_type ) : ?>
											<?php
											$display_row(
												esc_html__( 'Weight', 'redparts-sputnik' ),
												function( $product ) {
													return $product->get_weight();
												}
											);
											?>
										<?php elseif ( 'dimensions' === $row_type ) : ?>
											<?php
											$display_row(
												esc_html__( 'Dimensions', 'redparts-sputnik' ),
												function( $product ) {
													return wc_format_dimensions( $product->get_dimensions( false ) );
												}
											);
											?>
										<?php elseif ( 'attributes' === $row_type ) : ?>
											<?php foreach ( $attributes_list as $attribute_id => $attribute_name ) : ?>
												<?php
												$display_row(
													$attribute_name,
													function( $product ) use ( $attributes_values, $attribute_id ) {
														return $attributes_values[ $product->get_id() ][ $attribute_id ] ?? '';
													}
												);
												?>
											<?php endforeach; ?>
										<?php endif; ?>
									<?php endforeach; ?>

									<tr class="th-compare-table__row">
										<th class="th-compare-table__column th-compare-table__column--header"></th>
										<?php foreach ( $products as $product ) : ?>
											<td class="th-compare-table__column th-compare-table__column--product">
												<a
													class="th-compare-remove th-button th-button--style--secondary th-button--size--small"
													href="<?php echo esc_url( $this->get_remove_url( $product->get_id() ) ); ?>"
													data-product-id="<?php echo esc_attr( $product->get_id() ); ?>"
												>
													<?php esc_html_e( 'Remove', 'redparts-sputnik' ); ?>
												</a>
											</td>
										<?php endforeach; ?>
										<td class="th-compare-table__column th-compare-table__column--fake"></td>
									</tr>
									</tbody>
								</table>
							</div>
							<?php do_action( 'redparts_sputnik_compare_after_table' ); ?>
						</div>
					</div>
				<?php else : ?>
					<div class="th-block-empty th-block-empty--after-header">
						<div class="th-container">
							<div class="th-block-empty__body">
								<div class="th-block-empty__message">
									<?php
									echo wp_kses(
										__( 'You have no products in your compare list.<br>Go to the home page to start browsing our store.', 'redparts-sputnik' ), // SKIP-ESC.
										'redparts_sputnik_text'
									);
									?>
								</div>
								<div class="th-block-empty__action">
									<?php
									$shop_url = apply_filters( 'woocommerce_return_to_shop_redirect', wc_get_page_permalink( 'shop' ) );
									?>
									<a href="<?php echo esc_url( $shop_url ); ?>" class="th-button th-button--style--primary th-button--size--small">
										<?php esc_html_e( 'Return to shop', 'redparts-sputnik' ); ?>
									</a>
								</div>
							</div>
						</div>
					</div>
				<?php endif; ?>
			</div>
			<?php

			return ob_get_clean();
		}

		/**
		 * Outputs the button to adding to the comparison list.
		 *
		 * @noinspection PhpUnused
		 *
		 * @param integer|bool $product_id Product ID.
		 * @param string|array $classes    The list of classes that will be applied to the button.
		 */
		public function the_button( $product_id = false, $classes = '' ) {
			if ( ! $product_id ) {
				global $product;

				$product_id = ! is_null( $product ) && $product instanceof WC_Product ? $product->get_id() : 0;
			}

			if ( empty( $product_id ) ) {
				return;
			}

			$classes = apply_filters(
				'redparts_sputnik_compare_button_classes',
				redparts_sputnik_get_classes( 'th-compare-add', $classes )
			);

			ob_start();
			?>
			<a
				class="<?php echo esc_attr( implode( ' ', (array) $classes ) ); ?>"
				href="<?php echo esc_url( $this->get_add_url( $product_id ) ); ?>"
				data-product-id="<?php echo esc_attr( $product_id ); ?>"
			>
				<?php
				/**
				 * Hook: redparts_sputnik_compare_button_content.
				 *
				 * @hooked Compare::the_button_label - 100
				 * @hooked Compare::the_button_icon  - 200
				 */
				do_action( 'redparts_sputnik_compare_button_content' );
				?>
			</a>
			<?php

			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo apply_filters( 'redparts_sputnik_compare_the_button', ob_get_clean(), $product_id, $classes );
		}

		/**
		 * Outputs button label.
		 *
		 * @noinspection PhpUnused
		 */
		public function the_button_label() {
			?>
			<span class="th-compare-add__label">
				<?php
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo apply_filters(
					'redparts_sputnik_compare_button_label',
					esc_html__( 'Add to compare', 'redparts-sputnik' )
				);
				?>
			</span>
			<?php
		}

		/**
		 * Outputs button icon.
		 *
		 * @noinspection PhpUnused
		 */
		public function the_button_icon() {
			?>
			<span class="th-compare-add__icon">
				<?php
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo apply_filters( 'redparts_sputnik_compare_button_icon', '' );
				?>
			</span>
			<?php
		}

		/**
		 * Returns the URL to remove the product from the comparison list.
		 *
		 * @param int $product_id The identifier of the product to be deleted.
		 * @return string
		 */
		public function get_remove_url( int $product_id ): string {
			$args = array(
				'action'     => 'remove',
				'product_id' => $product_id,
			);

			$url = $this->get_page_url();

			if ( ! empty( $url ) ) {
				$url = esc_url_raw( add_query_arg( $args, $this->get_page_url() ) );
			}

			return apply_filters( 'redparts_sputnik_compare_get_remove_url', $url, $product_id );
		}

		/**
		 * Returns the URL to add to the comparison list.
		 *
		 * @param integer $product_id Product ID.
		 * @return mixed|void
		 */
		public function get_add_url( int $product_id ) {
			$args = array(
				'action'     => 'add',
				'product_id' => $product_id,
			);

			$url = $this->get_page_url();

			if ( ! empty( $url ) ) {
				$url = esc_url_raw( add_query_arg( $args, $this->get_page_url() ) );
			}

			return apply_filters( 'redparts_sputnik_compare_get_add_url', $url, $product_id );
		}

		/**
		 * Returns compare page URL.
		 *
		 * @return string
		 */
		public function get_page_url(): string {
			$page_id = Settings::instance()->get( 'compare_page' );
			$url     = $page_id ? get_page_link( $page_id ) : '';

			return apply_filters( 'redparts_sputnik_compare_get_page_url', $url );
		}

		/**
		 * Handles AJAX requests to add or remove products from the comparison list.
		 *
		 * @noinspection DuplicatedCode
		 */
		public function ajax() {
			if ( ! isset( $_POST['nonce'] ) || ! isset( $_POST['action'] ) ) {
				wp_send_json_error();
				wp_die();
			}

			WPML::switch_ajax_language();

			$nonce      = sanitize_key( wp_unslash( $_POST['nonce'] ) );
			$action     = sanitize_key( wp_unslash( $_POST['action'] ) );
			$product_id = null;

			if ( ! wp_verify_nonce( $nonce, $action ) ) {
				wp_send_json_error();
				wp_die();
			}

			if ( 'redparts_sputnik_compare_add' === $action ) {
				if ( ! isset( $_POST['product_id'] ) ) {
					wp_send_json_error();
					wp_die();
				}

				$product_id = WPML::get_original_product_id( absint( wp_unslash( $_POST['product_id'] ) ) );
				$product    = wc_get_product( $product_id );

				if ( ! $product ) {
					wp_send_json_error();
					wp_die();
				}

				if ( ! in_array( $product_id, $this->products_list, true ) ) {
					$this->products_list[] = $product_id;
				}
			} elseif ( 'redparts_sputnik_compare_remove' === $action ) {
				if ( ! isset( $_POST['product_id'] ) ) {
					wp_send_json_error();
					wp_die();
				}

				$product_id = WPML::get_original_product_id( absint( wp_unslash( $_POST['product_id'] ) ) );

				if ( in_array( $product_id, $this->products_list, true ) ) {
					$this->products_list = array_filter(
						$this->products_list,
						function( $item ) use ( $product_id ) {
							return $item !== $product_id;
						}
					);
				}
			} elseif ( 'redparts_sputnik_compare_clear' === $action ) {
				$this->products_list = array();
			} else {
				wp_send_json_error();
				wp_die();
			}

			setcookie( self::COOKIE_KEY, implode( ',', $this->products_list ), 0, COOKIEPATH, COOKIE_DOMAIN, false, false );

			$fragments = apply_filters(
				'redparts_sputnik_compare_fragments',
				array(
					'.th-block-compare' => $this->shortcode(),
				),
				$action,
				$product_id
			);

			wp_send_json(
				apply_filters(
					'redparts_sputnik_compare_ajax_response',
					array(
						'success'   => true,
						'count'     => count( $this->products_list ),
						'fragments' => $fragments,
					),
					$action,
					$product_id
				)
			);
			wp_die();
		}
	}
}
