<?php
/**
 * RedParts sputnik garage.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;
use WP_Term;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Garage' ) ) {
	/**
	 * Class Garage.
	 *
	 * @package RedParts\Sputnik
	 */
	class Garage extends Singleton {
		const VEHICLES_LIST_COOKIE_KEY = 'redparts_sputnik_garage';

		const CURRENT_VEHICLE_COOKIE_KEY = 'redparts_sputnik_current_vehicle';

		/**
		 * List of vehicle ids in garage.
		 *
		 * @var string[]
		 */
		protected $vehicles_list = array();

		/**
		 * Current vehicle.
		 *
		 * @var null|string
		 */
		protected $current_vehicle = null;

		/**
		 * Used to cache garage entities.
		 *
		 * @since 1.7.0
		 *
		 * @var array
		 */
		protected $cache = array();

		/**
		 * Initialization.
		 */
		public function init() {
			// The priority is 100 so that Redux has time to initialize.
			add_action( 'after_setup_theme', array( $this, 'deferred_init' ), 100 );
		}

		/**
		 * Deferred initialization.
		 */
		public function deferred_init() {
			if ( 'yes' !== Settings::instance()->get( 'autoparts_features' ) ) {
				return;
			}

			add_action( 'init', array( $this, 'load' ) );
			add_action( 'redparts_sputnik_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
			add_action( 'wp_footer', array( $this, 'the_vehicle_picker_modal' ) );

			add_filter( 'body_class', array( $this, 'body_class' ) );
			add_filter( 'woocommerce_product_query_tax_query', array( $this, 'product_query_tax_query' ), 10, 1 );
			add_filter( 'term_link', array( $this, 'product_category_link' ), 10, 3 );

			add_shortcode( 'redparts_sputnik_garage_add_button', array( $this, 'add_button_shortcode' ) );

			// Garage page.
			if ( 'yes' === Settings::instance()->get( 'garage_page' ) ) {
				add_filter( 'woocommerce_get_query_vars', array( $this, 'get_query_vars' ) );
				add_filter( 'woocommerce_endpoint_garage_title', array( $this, 'endpoint_garage_title' ) );
				add_filter( 'woocommerce_account_menu_items', array( $this, 'account_menu_items' ) );
				add_action( 'woocommerce_account_garage_endpoint', array( $this, 'page' ) );
			}

			if ( ! wp_doing_ajax() ) {
				return;
			}

			add_action( 'wp_ajax_redparts_sputnik_garage_add', array( $this, 'ajax_manage_garage' ) );
			add_action( 'wp_ajax_nopriv_redparts_sputnik_garage_add', array( $this, 'ajax_manage_garage' ) );

			add_action( 'wp_ajax_redparts_sputnik_garage_remove', array( $this, 'ajax_manage_garage' ) );
			add_action( 'wp_ajax_nopriv_redparts_sputnik_garage_remove', array( $this, 'ajax_manage_garage' ) );

			add_action( 'wp_ajax_redparts_sputnik_garage_set_current', array( $this, 'ajax_manage_garage' ) );
			add_action( 'wp_ajax_nopriv_redparts_sputnik_garage_set_current', array( $this, 'ajax_manage_garage' ) );
		}

		/**
		 * Enqueue scripts.
		 */
		public function enqueue_scripts() {
			wp_localize_script(
				'redparts-sputnik-garage',
				'redPartsSputnikGarageVars',
				array(
					'ajaxUrl'             => apply_filters( 'redparts_sputnik_ajax_url', '' ),
					'nonceAdd'            => wp_create_nonce( 'redparts_sputnik_garage_add' ),
					'nonceRemove'         => wp_create_nonce( 'redparts_sputnik_garage_remove' ),
					'nonceSetCurrent'     => wp_create_nonce( 'redparts_sputnik_garage_set_current' ),
					'attributeFilterName' => Vehicles::instance()->get_attribute_filter_name(),
				)
			);

			wp_enqueue_script( 'redparts-sputnik-garage' );
		}

		/**
		 * Adds the .th-garage-empty CSS class to the body as needed.
		 *
		 * @param string[] $classes Array of CSS classes.
		 *
		 * @return string[]
		 */
		public function body_class( array $classes ): array {
			if ( empty( $this->vehicles() ) ) {
				$classes[] = 'th-garage-empty';
			}

			return $classes;
		}

		/**
		 * Modifies product query.
		 *
		 * @param array $tax_query Array of tax queries.
		 *
		 * @return array
		 */
		public function product_query_tax_query( array $tax_query ): array {
			$attribute_slug = Vehicles::instance()->get_attribute_slug();

			foreach ( $tax_query as $index => $part ) {
				if ( ! is_array( $part ) || $part['taxonomy'] !== $attribute_slug || ! is_array( $part['terms'] ) ) {
					continue;
				}

				$tax_query[ $index ]['operator'] = 'IN';
				$tax_query[ $index ]['terms'][]  = Vehicles::instance()->get_all_term_slug();
			}

			return $tax_query;
		}

		/**
		 * Filter for **term_link** hook. Adds the "compatibility" query argument to the product category link.
		 *
		 * @since 1.7.0
		 *
		 * @param string  $termlink Term link URL.
		 * @param WP_Term $term     Term object.
		 * @param string  $taxonomy Taxonomy slug.
		 *
		 * @return string
		 * @noinspection PhpUnusedParameterInspection, PhpMissingReturnTypeInspection, PhpMissingParamTypeInspection
		 */
		public function product_category_link( $termlink, $term, $taxonomy ) {
			if ( 'product_cat' !== $taxonomy ) {
				return $termlink;
			}

			$filter_name = Vehicles::instance()->get_attribute_filter_name();

			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if ( isset( $_GET[ 'redparts_' . $filter_name ] ) ) {
				$termlink = add_query_arg( $filter_name, wc_clean( wp_unslash( $_GET[ 'redparts_' . $filter_name ] ) ), $termlink );
			} elseif ( isset( $_GET[ $filter_name ] ) ) {
				$termlink = add_query_arg( $filter_name, wc_clean( wp_unslash( $_GET[ $filter_name ] ) ), $termlink );
			} elseif ( isset( $_GET[ 'current_' . $filter_name ] ) ) {
				$termlink = add_query_arg( 'current_' . $filter_name, wc_clean( wp_unslash( $_GET[ 'current_' . $filter_name ] ) ), $termlink );
			}
			// phpcs:enable

			return $termlink;
		}

		/**
		 * Returns the current vehicle depends on the specified scope.
		 *
		 * @param string $scope Scope name.
		 *
		 * @return array|null
		 * @noinspection PhpMissingReturnTypeInspection
		 */
		public function get_current_vehicle( $scope = 'global' ) {
			$cache_key       = 'redparts_sputnik_garage_current_vehicle_' . $scope;
			$current_vehicle = $this->cache[ $cache_key ] ?? false;

			if ( false !== $current_vehicle ) {
				return $current_vehicle;
			}

			if ( 'shop' === $scope ) {
				$filter_name = Vehicles::instance()->get_attribute_filter_name();

				// phpcs:disable WordPress.Security.NonceVerification.Recommended
				if ( isset( $_GET[ 'redparts_' . $filter_name ] ) ) {
					$vehicle_instance_id = sanitize_text_field( wp_unslash( $_GET[ 'redparts_' . $filter_name ] ) );

					$current_vehicle = Vehicles::instance()->get_vehicle_by_instance_id( $vehicle_instance_id );
				} elseif ( isset( $_GET[ $filter_name ] ) ) {
					$vehicle_slug = sanitize_title( wp_unslash( $_GET[ $filter_name ] ) );

					$current_vehicle = Vehicles::instance()->get_vehicle_by( 'slug', $vehicle_slug );
				} elseif ( isset( $_GET[ 'current_' . $filter_name ] ) ) {
					$vehicle_instance_id = sanitize_text_field( wp_unslash( $_GET[ 'current_' . $filter_name ] ) );

					$current_vehicle = Vehicles::instance()->get_vehicle_by_instance_id( $vehicle_instance_id );
				}
				// phpcs:enable
			}

			if ( false === $current_vehicle && is_string( $this->current_vehicle ) ) {
				if ( preg_match( '#^[0-9]+$#', $this->current_vehicle ) ) {
					$current_vehicle = Vehicles::instance()->get_vehicle_by_id( absint( $this->current_vehicle ) );
				} elseif ( is_string( $this->current_vehicle ) ) {
					$current_vehicle = Vehicles::instance()->get_vehicle_by_instance_id( $this->current_vehicle );
				}
			}

			$current_vehicle = $current_vehicle ? $current_vehicle : null;

			$this->cache[ $cache_key ] = $current_vehicle;

			return $current_vehicle;
		}

		/**
		 * Handles garage AJAX requests.
		 */
		public function ajax_manage_garage() {
			if ( ! isset( $_POST['nonce'] ) ) {
				wp_send_json_error();
			}

			WPML::switch_ajax_language();

			$nonce = sanitize_key( wp_unslash( $_POST['nonce'] ) );

			if ( ! isset( $_POST['data']['vehicle_key'] ) ) {
				wp_send_json_error();
			}

			$vehicles    = $this->vehicles();
			$vehicle_key = sanitize_text_field( wp_unslash( $_POST['data']['vehicle_key'] ) );

			if ( empty( $vehicle_key ) ) {
				$vehicle = null;
			} else {
				$vehicle = Vehicles::instance()->get_vehicle_by_instance_id( $vehicle_key );
			}

			$response = array();

			if ( wp_verify_nonce( $nonce, 'redparts_sputnik_garage_add' ) ) {
				// Add.
				if ( ! $vehicle ) {
					wp_send_json_error(
						array(
							'message' => esc_html__( 'Specified vehicle not found', 'redparts-sputnik' ),
						)
					);
				}

				$vehicle_in_list = false;

				foreach ( $vehicles as $each_vehicle ) {
					$vehicle_in_list = $vehicle_in_list || $vehicle['instance_id'] === $each_vehicle['instance_id'];
				}

				if ( ! $vehicle_in_list ) {
					$vehicles[] = $vehicle;
				}
			} elseif ( wp_verify_nonce( $nonce, 'redparts_sputnik_garage_remove' ) ) {
				// Remove.
				if ( ! $vehicle ) {
					wp_send_json_error(
						array(
							'message' => esc_html__( 'Specified vehicle not found', 'redparts-sputnik' ),
						)
					);
				}

				$vehicles = array_filter(
					$vehicles,
					function ( $each_vehicle ) use ( $vehicle ) {
						if ( $vehicle['instance_id'] === $each_vehicle['instance_id'] ) {
							return false;
						}

						return true;
					}
				);

				$current_vehicle = $this->get_current_vehicle();

				if ( $current_vehicle && $current_vehicle['instance_id'] === $vehicle['instance_id'] ) {
					$this->current_vehicle = null;

					$response['set_current_vehicle_data'] = $this->get_current_vehicle_control_data( null );
				}
			} elseif ( wp_verify_nonce( $nonce, 'redparts_sputnik_garage_set_current' ) ) {
				// Set current.
				$vehicle_key = empty( $vehicle_key ) ? null : $vehicle_key;

				if ( null !== $vehicle_key ) {
					if ( ! $vehicle ) {
						wp_send_json_error();
					}

					$vehicle_in_list = false;

					foreach ( $vehicles as $each_vehicle ) {
						$vehicle_in_list = $vehicle_in_list || $vehicle['instance_id'] === $each_vehicle['instance_id'];
					}

					if ( ! $vehicle_in_list ) {
						wp_send_json_error();
					}
				}

				$this->current_vehicle = $vehicle_key;
			} else {
				wp_send_json_error();
			}

			$this->vehicles_list = array_map(
				function ( $each_vehicle ) {
					return $each_vehicle['instance_id'];
				},
				$vehicles
			);

			$this->save();
			$this->cache = array();

			if ( ! empty( $_POST['data']['badges'] ) ) {
				$response['compatibility_badges'] = $this->get_compatibility_badges(
					sanitize_text_field( wp_unslash( $_POST['data']['badges'] ) )
				);
			}

			$response['fragments'] = apply_filters(
				'redparts_sputnik_garage_fragments',
				array(
					'.th-garage'                 => $this->get_vehicles_list_template(),
					'.th-vehicles-list--current' => $this->get_current_vehicle_control(),
				)
			);

			wp_send_json_success(
				array_merge(
					$response,
					array(
						'vehicles'        => $this->vehicles_list,
						'current_vehicle' => $this->current_vehicle,
					)
				)
			);
		}

		/**
		 * Returns compatibility badges by product ids.
		 *
		 * @param string $product_ids Comma separated product ids.
		 *
		 * @return array
		 */
		public function get_compatibility_badges( string $product_ids ): array {
			$compatibility_badges = array();

			$product_ids = explode( ',', $product_ids );
			$product_ids = array_map( 'absint', (array) array_unique( $product_ids ) );
			$product_ids = array_slice( $product_ids, 0, 250 );

			foreach ( $product_ids as $product_id ) {
				ob_start();

				$this->the_compatibility_badge( $product_id );

				$compatibility_badges[ $product_id ] = ob_get_clean();
			}

			return $compatibility_badges;
		}

		/**
		 * Returns query vars.
		 *
		 * @param array $query_vars Query vars.
		 *
		 * @return array
		 */
		public function get_query_vars( array $query_vars ): array {
			$query_vars['garage'] = 'garage';

			return $query_vars;
		}

		/**
		 * Returns garage endpoint title.
		 *
		 * @return string
		 */
		public function endpoint_garage_title(): string {
			return esc_html__( 'Garage', 'redparts-sputnik' );
		}

		/**
		 * Adds garage to the account menu.
		 *
		 * @param array $items Account menu items.
		 *
		 * @return array
		 */
		public function account_menu_items( array $items ): array {
			$after = apply_filters( 'redparts_sputnik_garage_account_menu_item_after', 'dashboard' );
			$index = array_search( $after, array_keys( $items ), true );
			$index = false !== $index ? $index : -1;

			$result  = array_slice( $items, 0, $index + 1, true );
			$result += array( 'garage' => $this->endpoint_garage_title() );
			$result += array_slice( $items, $index + 1, count( $items ) - $index - 1, true );

			return $result;
		}

		/**
		 * Returns the vehicles are on a list.
		 *
		 * @return array
		 * @noinspection PhpUnused
		 */
		public function vehicles(): array {
			$cache_key = 'redparts_sputnik_garage_vehicles_list';
			$vehicles  = $this->cache[ $cache_key ] ?? false;

			if ( false === $vehicles ) {
				$vehicles = array();

				foreach ( $this->vehicles_list as $vehicles_list_item ) {
					if ( preg_match( '#^[0-9]+$#', $vehicles_list_item ) ) {
						$vehicle = Vehicles::instance()->get_vehicle_by_id( absint( $vehicles_list_item ) );
					} else {
						$vehicle = Vehicles::instance()->get_vehicle_by_instance_id( $vehicles_list_item );
					}

					$vehicles[] = $vehicle;
				}

				$vehicles = array_filter(
					$vehicles,
					function( $item ) {
						return null !== $item;
					}
				);

				$this->cache[ $cache_key ] = $vehicles;
			}

			return $vehicles;
		}

		/**
		 * Returns vehicles list template.
		 *
		 * @return string
		 */
		public function get_vehicles_list_template(): string {
			ob_start();

			$this->the_vehicles_list_template();

			return ob_get_clean();
		}

		/**
		 * Outputs vehicles list.
		 */
		public function the_vehicles_list_template() {
			$vehicles = $this->vehicles();

			?>
			<div class="th-garage">
				<?php if ( 0 !== count( $vehicles ) ) : ?>
					<div class="th-card__header">
						<h5><?php echo esc_html__( 'Garage', 'redparts-sputnik' ); ?></h5>
					</div>
					<div class="th-card__divider"></div>
					<div class="th-card__body">
						<div class="th-vehicles-list th-vehicles-list--location--account">
							<div class="th-vehicles-list__body">
								<?php foreach ( $vehicles as $vehicle ) : ?>
									<div class="th-vehicles-list__item">
										<div class="th-vehicles-list__item-info">
											<div class="th-vehicles-list__item-name">
												<?php echo esc_html( Vehicles::get_vehicle_name( $vehicle ) ); ?>
											</div>
											<?php $description = Vehicles::get_vehicle_description( $vehicle ); ?>
											<?php if ( ! empty( $description ) ) : ?>
												<div class="th-vehicles-list__item-details">
													<?php echo esc_html( $description ); ?>
												</div>
											<?php endif; ?>
											<div class="th-vehicles-list__item-links">
												<a href="<?php echo esc_url( Vehicles::get_vehicle_link( $vehicle ) ); ?>">
													<?php echo esc_html__( 'Show Parts', 'redparts-sputnik' ); ?>
												</a>
											</div>
										</div>
										<button
											type="button"
											class="th-vehicles-list__item-remove"
											<?php
											$this->the_remove_button_attrs(
												array(
													'vehicle_key'   => $vehicle['instance_id'],
													'loading_class' => 'th-vehicles-list__item-remove--loading',
												)
											);
											?>
										>
											<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16">
												<path d="M2 4V2h3V1h6v1h3v2H2zm11 9c0 1.1-.9 2-2 2H5c-1.1 0-2-.9-2-2V5h10v8z"></path>
											</svg>
										</button>
									</div>
								<?php endforeach; ?>
							</div>
						</div>
					</div>
					<div class="th-card__divider"></div>
				<?php endif; ?>
			</div>
			<?php
		}

		/**
		 * Outputs garage page content.
		 */
		public function page() {
			?>
			<div class="th-card th-card--padding--2">
				<?php
				$this->the_vehicles_list_template();
				?>

				<div class="th-card__header">
					<h5><?php echo esc_html__( 'Add A Vehicle', 'redparts-sputnik' ); ?></h5>
				</div>
				<div class="th-card__divider"></div>
				<div class="th-card__body">
					<form class="th-garage-add-form">
						<?php echo do_shortcode( '[redparts_sputnik_vehicle_form location="account"]' ); ?>

						<div class="th-mt-4 th-pt-2">
							<?php
							echo do_shortcode(
								'[redparts_sputnik_garage_add_button
									class="th-button th-button--style--primary th-button--size--normal"
									loading_class="th-button--loading"
								]'
							);
							?>
						</div>
					</form>
				</div>
			</div>
			<?php
		}

		/**
		 * Returns vehicle data for checkbox.
		 *
		 * @param array|null $vehicle Vehicle.
		 *
		 * @return array
		 * @noinspection PhpMissingParamTypeInspection
		 */
		public function get_current_vehicle_control_data( $vehicle ): array {
			if ( $vehicle ) {
				$placeholder = sprintf(
					apply_filters(
						'redparts_sputnik_search_placeholder',
						// translators: %s vehicle name.
						esc_html__( 'Search for %s', 'redparts-sputnik' )
					),
					Vehicles::get_vehicle_name( $vehicle )
				);

				return array(
					'vehicle_key'            => $vehicle['instance_id'],
					'search_placeholder'     => esc_html( $placeholder ),
					'attribute_filter_value' => Vehicles::instance()->get_attribute_filter_value( $vehicle ),
				);
			}

			$placeholder = apply_filters(
				'redparts_sputnik_search_default_placeholder',
				esc_html__( 'Enter Keyword or Part Number', 'redparts-sputnik' )
			);

			return array(
				'vehicle_key'            => '',
				'search_placeholder'     => esc_html( $placeholder ),
				'attribute_filter_value' => '',
			);
		}

		/**
		 * Returns template of the control of the current vehicle.
		 *
		 * @return string
		 */
		public function get_current_vehicle_control(): string {
			ob_start();

			$this->the_current_vehicle_control();

			return ob_get_clean();
		}

		/**
		 * Outputs template of the control of the current vehicle.
		 *
		 * @param array $args Arguments.
		 */
		public function the_current_vehicle_control( $args = array() ) {
			$args = wp_parse_args(
				$args,
				array(
					'class'    => '',
					'location' => '',
				)
			);

			$classes = array(
				'th-vehicles-list th-vehicles-list--current',
			);

			if ( ! empty( $args['class'] ) ) {
				$classes[] = $args['class'];
			}
			if ( ! empty( $args['location'] ) ) {
				$classes[] = 'th-vehicles-list--location--' . $args['location'];
			}

			$current_vehicle = $this->get_current_vehicle();

			?>
			<div class="<?php echo esc_attr( join( ' ', $classes ) ); ?>">
				<form class="th-vehicles-list__body">
					<div class="th-vehicles-list__item">
						<label class="th-vehicles-list__item-label">
							<input
								type="radio"
								name="th-header-vehicle"
								class="th-vehicles-list__item-radio"
								data-th-current-vehicle="<?php echo esc_attr( wp_json_encode( $this->get_current_vehicle_control_data( null ) ) ); ?>"
								<?php checked( null === $current_vehicle ); ?>
							>
							<span class="th-vehicles-list__item-info">
								<span class=" th-vehicles-list__item-name">
									<?php echo esc_html__( 'All Vehicles', 'redparts-sputnik' ); ?>
								</span>
							</span>
						</label>
					</div>
					<?php foreach ( $this->vehicles() as $vehicle ) : ?>
						<div class="th-vehicles-list__item">
							<label class="th-vehicles-list__item-label">
								<input
									type="radio"
									name="th-header-vehicle"
									class="th-vehicles-list__item-radio"
									data-th-current-vehicle="<?php echo esc_attr( wp_json_encode( $this->get_current_vehicle_control_data( $vehicle ) ) ); ?>"
									<?php checked( $current_vehicle && $vehicle['instance_id'] === $current_vehicle['instance_id'] ); ?>
								>
								<span class="th-vehicles-list__item-info">
									<span class="th-vehicles-list__item-name">
										<?php echo esc_html( Vehicles::get_vehicle_name( $vehicle ) ); ?>
									</span>
									<?php $description = Vehicles::get_vehicle_description( $vehicle ); ?>
									<?php if ( ! empty( $description ) ) : ?>
										<span class="th-vehicles-list__item-details">
											<?php echo esc_html( $description ); ?>
										</span>
									<?php endif; ?>
								</span>
							</label>
							<button
								type="button"
								class="th-vehicles-list__item-remove"
								<?php
								$this->the_remove_button_attrs(
									array(
										'vehicle_key'   => $vehicle['instance_id'],
										'loading_class' => 'th-vehicles-list__item-remove--loading',
									)
								);
								?>
							>
								<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16">
									<path d="M2 4V2h3V1h6v1h3v2H2zm11 9c0 1.1-.9 2-2 2H5c-1.1 0-2-.9-2-2V5h10v8z"></path>
								</svg>
							</button>
						</div>
					<?php endforeach; ?>
				</form>
			</div>
			<?php
		}

		/**
		 * Shortcode that returns the add vehicle button.
		 *
		 * @param array $args Arguments.
		 *
		 * @return string
		 */
		public function add_button_shortcode( $args = array() ): string {
			$args = shortcode_atts(
				array(
					'type'          => 'submit',
					'title'         => '',
					'vehicle_id'    => '',
					'class'         => '',
					'loading_class' => '',
				),
				$args
			);

			$classes = array(
				'th-garage-add',
			);

			if ( ! empty( $args['class'] ) ) {
				$classes[] = $args['class'];
			}

			$title = esc_html__( 'Add A Vehicle', 'redparts-sputnik' );

			if ( ! empty( $args['title'] ) ) {
				$title = $args['title'];
			}

			ob_start();

			?>
			<button
				type="<?php echo esc_attr( $args['type'] ); ?>"
				class="<?php echo esc_attr( join( ' ', $classes ) ); ?>"
				data-vehicle-id="<?php echo esc_attr( $args['vehicle_id'] ); ?>"
				data-loading-class="<?php echo esc_attr( $args['loading_class'] ); ?>"
				<?php disabled( empty( $args['vehicle_id'] ) ); ?>
			>
				<?php echo esc_html( $title ); ?>
			</button>
			<?php

			return ob_get_clean();
		}

		/**
		 * Output attributes required for the remove vehicle button.
		 *
		 * @param array|string $args Attributes.
		 */
		public function the_remove_button_attrs( $args = array() ) {
			$args = wp_parse_args(
				$args,
				array(
					'vehicle_key'   => '',
					'loading_class' => '',
				)
			)

			?>
			data-th-garage-remove
			data-th-vehicle-key="<?php echo esc_attr( $args['vehicle_key'] ); ?>"
			data-th-loading-class="<?php echo esc_attr( $args['loading_class'] ); ?>"
			<?php
		}

		/**
		 * Outputs vehicle picker modal.
		 *
		 * @noinspection PhpUnused
		 */
		public function the_vehicle_picker_modal() {
			$vehicles      = $this->vehicles();
			$current_panel = 0 !== count( $vehicles ) ? 'list' : 'form';

			?>
			<div class="th-modal th-modal--size--md" tabindex="-1" role="dialog">
				<div class="th-modal__backdrop"></div>
				<div class="th-modal__container">
					<div class="th-modal__wrapper">
						<div class="th-modal__window">
							<button class="th-modal__close" type="button">
								<?php
								// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
								echo apply_filters(
									'redparts_sputnik_modal_close_button_content',
									esc_html__( 'Close', 'redparts-sputnik' )
								);
								?>
							</button>


							<div class="th-vehicle-picker-modal">
								<?php
								$classes = array(
									'th-vehicle-picker-modal__panel',
								);

								if ( 'list' === $current_panel ) {
									$classes[] = 'th-vehicle-picker-modal__panel--active';
								}
								?>
								<div class="<?php redparts_sputnik_the_classes( ...$classes ); ?>" data-panel="list">
									<div class="th-vehicle-picker-modal__header">
										<?php echo esc_html__( 'Select Vehicle', 'redparts-sputnik' ); ?>
									</div>
									<div class="th-vehicle-picker-modal__body">
										<?php
										$this->the_current_vehicle_control(
											array(
												'location' => 'modal',
												'class'    => 'th-vehicle-picker-modal__list',
											)
										);
										?>

										<button
											type="button"
											data-to-panel="form"
											class="th-button th-button--style--secondary th-button--size--small th-mt-2 th-vehicle-picker-modal__add-button"
										>
											<?php echo esc_html__( 'Add Vehicle', 'redparts-sputnik' ); ?>
										</button>
									</div>
									<div class="th-vehicle-picker-modal__footer">
										<button type="button" class="th-button th-button--style--secondary th-button--size--small th-vehicle-picker-modal__close-button">
											<?php echo esc_html__( 'Cancel', 'redparts-sputnik' ); ?>
										</button>
										<button type="button" class="th-button th-button--style--primary th-button--size--small th-vehicle-picker-modal__select-button">
											<?php echo esc_html__( 'Select Vehicle', 'redparts-sputnik' ); ?>
										</button>
									</div>
								</div>
								<?php
								$classes = array(
									'th-garage-add-form',
									'th-vehicle-picker-modal__panel',
								);

								if ( 'form' === $current_panel ) {
									$classes[] = 'th-vehicle-picker-modal__panel--active';
								}
								?>
								<div class="<?php redparts_sputnik_the_classes( ...$classes ); ?>" data-panel="form">
									<div class="th-vehicle-picker-modal__header">
										<?php echo esc_html__( 'Add A Vehicle', 'redparts-sputnik' ); ?>
									</div>
									<div class="th-vehicle-picker-modal__body">
										<?php echo do_shortcode( '[redparts_sputnik_vehicle_form location="modal"]' ); ?>
									</div>
									<div class="th-vehicle-picker-modal__footer">
										<button type="button" class="th-button th-button--style--secondary th-button--size--small th-vehicle-picker-modal__back-to-list" data-to-panel="list">
											<?php echo esc_html__( 'Back to list', 'redparts-sputnik' ); ?>
										</button>

										<?php
										echo do_shortcode(
											'[redparts_sputnik_garage_add_button
												class="th-button th-button--style--primary th-button--size--small"
											 	loading_class="th-button--loading"
											]'
										);
										?>
									</div>
								</div>
							</div>
						</div>
					</div>
				</div>
			</div>
			<?php
		}

		/**
		 * Outputs compatibility badge.
		 *
		 * @param int   $product_id Product ID.
		 * @param array $args       Additional arguments.
		 */
		public function the_compatibility_badge( int $product_id, $args = array() ) {
			$args = wp_parse_args(
				$args,
				array(
					'scope' => 'global',
					'class' => '',
				)
			);

			$attribute_slug  = Vehicles::instance()->get_attribute_slug();
			$current_vehicle = $this->get_current_vehicle( $args['scope'] );

			if ( ! $attribute_slug ) {
				return;
			}

			$attribute_values = wc_get_product_terms( WPML::get_original_product_id( $product_id ), $attribute_slug, array( 'fields' => 'all' ) );
			$attribute_values = array_map(
				function( $item ) {
					return array(
						'id'   => $item->term_id,
						'slug' => $item->slug,
					);
				},
				$attribute_values
			);

			$all_term_slug = Vehicles::instance()->get_all_term_slug();

			$attribute_value_slugs = array_map(
				function( $item ) {
					return $item['slug'];
				},
				$attribute_values
			);

			$mode    = 'disabled';
			$title   = '';
			$classes = array(
				'th-status-badge',
				'th-status-badge--has-icon',
				'th-status-badge--has-text',
			);

			if ( $current_vehicle ) {
				$any_fit = false;
				$all_fit = true;

				if ( is_array( $current_vehicle['slug'] ) ) {
					foreach ( $current_vehicle['slug'] as $slug ) {
						$any_fit = $any_fit || in_array( $slug, $attribute_value_slugs, true );
						$all_fit = $all_fit && in_array( $slug, $attribute_value_slugs, true );
					}
				}

				$vehicle_name_contains_years_range = Vehicles::is_vehicle_name_contains_years_range();

				if ( empty( $attribute_values )
					|| ( $vehicle_name_contains_years_range && is_array( $current_vehicle['slug'] ) && $any_fit && ! $all_fit )
				) {
					$mode  = 'may_not_fit';
					$title = sprintf(
						// translators: %s vehicle name.
						esc_html__( 'May Not Fit for %s', 'redparts-sputnik' ),
						Vehicles::get_vehicle_name( $current_vehicle )
					);
					$classes[] = 'th-status-badge--style--warning';
				} elseif ( in_array( $current_vehicle['slug'], $attribute_value_slugs, true )
					|| ( ! $vehicle_name_contains_years_range && $any_fit )
					|| ( $vehicle_name_contains_years_range && is_array( $current_vehicle['slug'] ) && $all_fit )
				) {
					$mode  = 'fit';
					$title = sprintf(
						// translators: %s vehicle name.
						esc_html__( 'Fit for %s', 'redparts-sputnik' ),
						Vehicles::get_vehicle_name( $current_vehicle )
					);
					$classes[] = 'th-status-badge--style--success';
				} elseif ( in_array( $all_term_slug, $attribute_value_slugs, true ) ) {
					$mode      = 'fit_all';
					$title     = esc_html__( 'Fit for All Vehicles', 'redparts-sputnik' );
					$classes[] = 'th-status-badge--style--unknown';
				} else {
					$mode  = 'not_fit';
					$title = sprintf(
						// translators: %s vehicle name.
						esc_html__( 'Not Fit for %s', 'redparts-sputnik' ),
						Vehicles::get_vehicle_name( $current_vehicle )
					);
					$classes[] = 'th-status-badge--style--failure';
				}
			}

			$badge_id = sprintf( '%s:%s', $args['scope'], $product_id );

			?>
			<div
				<?php if ( ! empty( $args['class'] ) ) : ?>
					class="<?php echo esc_attr( implode( ' ', (array) $args['class'] ) ); ?>"
				<?php endif; ?>
				data-th-compatibility-badge="<?php echo esc_attr( $badge_id ); ?>"
			>
				<?php if ( 'disabled' !== $mode ) : ?>
					<div class="<?php echo esc_attr( implode( ' ', $classes ) ); ?>">
						<div class="th-status-badge__body">
							<div class="th-status-badge__icon">
								<?php if ( in_array( $mode, array( 'fit', 'fit_all' ), true ) ) : ?>
									<svg width="13" height="13">
										<path d="M12,4.4L5.5,11L1,6.5l1.4-1.4l3.1,3.1L10.6,3L12,4.4z"></path>
									</svg>
								<?php else : ?>
									<svg xmlns="http://www.w3.org/2000/svg" width="13" height="13">
										<path d="M6.5 0C2.9 0 0 2.9 0 6.5S2.9 13 6.5 13 13 10.1 13 6.5 10.1 0 6.5 0zm0 2c.9 0 1.7.3 2.4.7L2.7 8.9C2.3 8.2 2 7.4 2 6.5 2 4 4 2 6.5 2zm0 9c-.9 0-1.7-.3-2.4-.7l6.2-6.2c.4.7.7 1.5.7 2.4C11 9 9 11 6.5 11z"></path>
									</svg>
								<?php endif; ?>
							</div>
							<div class="th-status-badge__text"><?php echo esc_html( $title ); ?></div>
							<div
								class="th-status-badge__tooltip"
								tabindex="0"
								data-tippy-content="<?php echo esc_attr( $title ); ?>"
							></div>
						</div>
					</div>
				<?php endif; ?>
			</div>
			<?php
		}

		/**
		 * Loads the vehicles list and the current vehicle.
		 */
		public function load() {
			// Vehicles list.
			if ( is_user_logged_in() ) {
				$vehicles_list = get_user_meta( get_current_user_id(), self::VEHICLES_LIST_COOKIE_KEY, true );

				if ( ! empty( $vehicles_list ) ) {
					$this->vehicles_list = $vehicles_list;
				}
			} elseif ( isset( $_COOKIE[ self::VEHICLES_LIST_COOKIE_KEY ] ) ) {
				$vehicles_list = sanitize_text_field( wp_unslash( $_COOKIE[ self::VEHICLES_LIST_COOKIE_KEY ] ) );

				$this->vehicles_list = explode( ',', $vehicles_list );
			}

			$this->vehicles_list = array_map(
				function ( $item ) {
					return (string) $item;
				},
				$this->vehicles_list
			);

			// Current vehicle.
			$vehicle_key = null;

			if ( is_user_logged_in() ) {
				$vehicle_key = (string) get_user_meta( get_current_user_id(), self::CURRENT_VEHICLE_COOKIE_KEY, true );
			} elseif ( isset( $_COOKIE[ self::CURRENT_VEHICLE_COOKIE_KEY ] ) ) {
				$vehicle_key = sanitize_text_field( wp_unslash( $_COOKIE[ self::CURRENT_VEHICLE_COOKIE_KEY ] ) );
			}

			$this->current_vehicle = $vehicle_key;
		}

		/**
		 * Saves the vehicles list and the current vehicle.
		 */
		public function save() {
			if ( is_user_logged_in() ) {
				update_user_meta( get_current_user_id(), self::VEHICLES_LIST_COOKIE_KEY, $this->vehicles_list );
				update_user_meta( get_current_user_id(), self::CURRENT_VEHICLE_COOKIE_KEY, $this->current_vehicle );
			} else {
				setcookie( self::VEHICLES_LIST_COOKIE_KEY, implode( ',', $this->vehicles_list ), 0, COOKIEPATH, COOKIE_DOMAIN, false, false );
				setcookie( self::CURRENT_VEHICLE_COOKIE_KEY, $this->current_vehicle, 0, COOKIEPATH, COOKIE_DOMAIN, false, false );
			}
		}
	}
}
