<?php
/**
 * This file contains helpers and other code used in the admin panel only.
 *
 * @package RedParts\Sputnik
 * @since 1.9.0
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;
use stdClass;
use WC_Product_Attribute;
use WP_Error;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Lazy_Select' ) ) {
	/**
	 * Class Lazy_Select
	 *
	 * @since 1.9.0
	 */
	class Lazy_Select extends Singleton {
		/**
		 * Initialization.
		 *
		 * @since 1.9.0
		 */
		public function init() {
			add_action( 'init', array( $this, 'deferred_init' ) );
			add_filter( 'product_attributes_type_selector', array( $this, 'get_attribute_types' ) );
			add_filter( 'woocommerce_product_option_terms', array( $this, 'product_terms_selection_field' ), 10, 3 );

			if ( wp_doing_ajax() ) {
				add_action( 'wp_ajax_redparts_sputnik_lazy_select_terms', array( $this, 'ajax' ) );
				add_action( 'wp_ajax_nopriv_redparts_sputnik_lazy_select_terms', array( $this, 'ajax' ) );
			}

		}

		/**
		 * Deferred initialization.
		 *
		 * @since 1.9.0
		 */
		public function deferred_init() {

		}

		/**
		 * Handles AJAX requests.
		 *
		 * @since 1.9.0
		 */
		public function ajax() {
			WPML::switch_ajax_language();

			// phpcs:disable WordPress.Security.NonceVerification.Recommended
			if ( ! isset( $_GET['attribute_id'] ) ) {
				wp_die( -1 );
			}

			$s            = sanitize_text_field( wp_unslash( $_GET['s'] ?? '' ) );
			$attribute_id = absint( wp_unslash( $_GET['attribute_id'] ) );
			$attribute    = wc_get_attribute( $attribute_id );
			// phpcs:enable

			if ( ! $attribute ) {
				wp_die( -1 );
			}

			$args = array(
				'orderby'    => ! empty( $attribute_taxonomy->attribute_orderby ) ? $attribute_taxonomy->attribute_orderby : 'name',
				'hide_empty' => 0,
				'number'     => 20,
			);

			if ( ! empty( $s ) ) {
				$args['search'] = $s;
			}

			$result = get_terms( $attribute->slug, apply_filters( 'woocommerce_product_attribute_terms', $args ) );

			if ( $result instanceof WP_Error ) {
				wp_die( -1 );
			}

			$result = array_map(
				function( $term ) {
					return array(
						'id'   => $term->term_id,
						'text' => $term->name,
					);
				},
				$result
			);

			wp_send_json_success( $result );
		}

		/**
		 * Returns an associative array of attribute types.
		 *
		 * @since 1.9.0
		 *
		 * @param array $types An associative array of attribute types.
		 *
		 * @return array
		 */
		public function get_attribute_types( array $types = array() ): array {
			return array_merge( $types, array( 'lazy_select' => esc_html__( 'Lazy select', 'redparts-sputnik' ) ) );
		}

		/**
		 * Outputs terms selection field.
		 *
		 * @since 1.9.0
		 *
		 * @param stdClass             $attribute_taxonomy Attribute taxonomy.
		 * @param int                  $i                  Attribute index.
		 * @param WC_Product_Attribute $attribute          Attribute object.
		 */
		public function product_terms_selection_field( stdClass $attribute_taxonomy, int $i, WC_Product_Attribute $attribute ) {
			$types = array_keys( $this->get_attribute_types() );

			if ( ! in_array( $attribute_taxonomy->attribute_type, $types, true ) ) {
				return;
			}

			$options = $attribute->get_options();
			$options = ! empty( $options ) ? $options : array();

			?>
			<!--suppress HtmlFormInputWithoutLabel -->
			<select
				multiple="multiple"
				class="redparts-sputnik-lazy-select"
				name="attribute_values[<?php echo esc_attr( $i ); ?>][]"
				data-placeholder="<?php esc_attr_e( 'Select terms', 'redparts-sputnik' ); ?>"
				data-ajax-url="<?php echo esc_url( apply_filters( 'redparts_sputnik_ajax_url', '' ) ); ?>"
				data-attribute-id="<?php echo esc_attr( $attribute->get_id() ); ?>"
			>
				<?php
				if ( ! empty( $options ) ) {
					$args  = array(
						'orderby'    => ! empty( $attribute_taxonomy->attribute_orderby ) ? $attribute_taxonomy->attribute_orderby : 'name',
						'hide_empty' => 0,
						'include'    => $options,
					);
					$terms = get_terms( $attribute->get_taxonomy(), apply_filters( 'woocommerce_product_attribute_terms', $args ) );

					if ( $terms ) {
						foreach ( $terms as $term ) {
							?>
							<option
								value="<?php echo esc_attr( $term->term_id ); ?>"
								<?php
								// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Already escaped in wc_selected.
								echo wc_selected( $term->term_id, $options );
								?>
							>
								<?php echo esc_attr( apply_filters( 'woocommerce_product_attribute_term_name', $term->name, $term ) ); ?>
							</option>
							<?php
						}
					}
				}
				?>
			</select>
			<?php
		}
	}
}
