<?php
/**
 * This file contains code related to the quickview.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

use RedParts\Sputnik\WPML\WPML;
use WC_Product;
use WP_Query;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Quickview' ) ) {
	/**
	 * Class Quickview
	 */
	class Quickview extends Singleton {
		/**
		 * Initialization.
		 */
		public function init() {
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ), 20 );
			add_action( 'wp_footer', array( $this, 'the_placeholder' ) );

			if ( wp_doing_ajax() ) {
				add_action( 'wp_ajax_redparts_sputnik_quickview_load', array( $this, 'ajax' ) );
				add_action( 'wp_ajax_nopriv_redparts_sputnik_quickview_load', array( $this, 'ajax' ) );
			}

			// Button content.
			add_action( 'redparts_sputnik_quickview_button_content', array( $this, 'the_button_label' ), 100 );
			add_action( 'redparts_sputnik_quickview_button_content', array( $this, 'the_button_icon' ), 200 );

			// Quickview product.
			add_action( 'redparts_sputnik_quickview_product', array( $this, 'sale_badge' ), 100 );

			// Quickview gallery.
			add_action( 'redparts_sputnik_quickview_gallery', 'woocommerce_show_product_images', 100 );

			// Quickview summary.
			add_action( 'redparts_sputnik_quickview_summary', 'woocommerce_template_single_title', 100 );
			add_action( 'redparts_sputnik_quickview_summary', 'woocommerce_template_single_rating', 200 );
			add_action( 'redparts_sputnik_quickview_summary', 'woocommerce_template_single_meta', 300 );
			add_action( 'redparts_sputnik_quickview_summary', 'woocommerce_template_single_excerpt', 400 );
			add_action( 'redparts_sputnik_quickview_summary', 'woocommerce_template_single_price', 500 );
			add_action( 'redparts_sputnik_quickview_summary', array( $this, 'stock_badge' ), 600 );
			add_action( 'redparts_sputnik_quickview_summary', 'woocommerce_template_single_add_to_cart', 700 );
		}

		/**
		 * Enqueue scripts.
		 */
		public function enqueue_scripts() {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return;
			}

			// WooCommerce scripts and styles.
			wp_enqueue_script( 'wc-add-to-cart-variation' );

			if ( current_theme_supports( 'wc-product-gallery-zoom' ) ) {
				wp_enqueue_script( 'zoom' );
			}
			if ( current_theme_supports( 'wc-product-gallery-slider' ) ) {
				wp_enqueue_script( 'flexslider' );
			}
			if ( current_theme_supports( 'wc-product-gallery-lightbox' ) ) {
				wp_enqueue_script( 'photoswipe-ui-default' );
				wp_enqueue_style( 'photoswipe-default-skin' );
				add_action( 'wp_footer', 'woocommerce_photoswipe' );
			}
			wp_enqueue_script( 'wc-single-product' );

			wp_register_script(
				'redparts-sputnik-quickview',
				Plugin::instance()->url( 'assets/js/quickview.js' ),
				array(
					'jquery',
					'wc-add-to-cart-variation',
					'zoom',
					'flexslider',
					'photoswipe-ui-default',
					'wc-single-product',
				),
				Plugin::VERSION,
				true
			);
			wp_localize_script(
				'redparts-sputnik-quickview',
				'redPartsSputnikQuickviewVars',
				array(
					'ajaxUrl' => apply_filters( 'redparts_sputnik_ajax_url', '' ),
					'nonce'   => wp_create_nonce( 'redparts_sputnik_quickview_load' ),
				)
			);

			wp_enqueue_script( 'redparts-sputnik-quickview' );
		}

		/**
		 * Outputs a placeholder for the quickview modal.
		 *
		 * @noinspection PhpUnused
		 */
		public function the_placeholder() {
			if ( ! class_exists( 'WooCommerce' ) ) {
				return;
			}

			?>
			<div class="th-quickview" tabindex="-1" role="dialog">
				<div class="th-quickview__backdrop"></div>
				<div class="th-quickview__container">
					<div class="th-quickview__wrapper">
						<div class="th-quickview__window"></div>
					</div>
				</div>
			</div>
			<?php
		}

		/**
		 * Outputs the button that opens a quickview.
		 *
		 * @noinspection PhpUnused
		 *
		 * @param integer|bool $product_id Product ID.
		 * @param string|array $classes    The list of classes that will be applied to the button.
		 */
		public function the_button( $product_id = false, $classes = '' ) {
			if ( ! $product_id ) {
				global $product;

				$product_id = ! is_null( $product ) && $product instanceof WC_Product ? $product->get_id() : 0;
			}

			if ( empty( $product_id ) ) {
				return;
			}

			$classes = apply_filters(
				'redparts_sputnik_quickview_button_classes',
				redparts_sputnik_get_classes( 'th-quickview-open', $classes )
			);

			ob_start();
			?>
			<button
				type="button"
				class="<?php echo esc_attr( implode( ' ', (array) $classes ) ); ?>"
				data-product-id="<?php echo esc_attr( $product_id ); ?>"
			>
				<?php
				/**
				 * Hook: redparts_sputnik_quickview_button_content.
				 *
				 * @hooked Quickview::the_button_label - 100
				 * @hooked Quickview::the_button_icon  - 200
				 */
				do_action( 'redparts_sputnik_quickview_button_content' );
				?>
			</button>
			<?php

			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo apply_filters( 'redparts_sputnik_quickview_the_button', ob_get_clean(), $product_id, $classes );
		}

		/**
		 * Outputs button label.
		 *
		 * @noinspection PhpUnused
		 */
		public function the_button_label() {
			?>
			<span class="th-quickview-open__label">
				<?php
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo apply_filters(
					'redparts_sputnik_quickview_button_label',
					esc_html__( 'Quick View', 'redparts-sputnik' )
				);
				?>
			</span>
			<?php
		}

		/**
		 * Outputs button icon.
		 */
		public function the_button_icon() {
			?>
			<span class="th-quickview-open__icon">
				<?php
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				echo apply_filters( 'redparts_sputnik_quickview_button_icon', '' );
				?>
			</span>
			<?php
		}

		/**
		 * Outputs quickview content.
		 *
		 * @param int $product_id Product id.
		 */
		public function the_content( int $product_id ) {
			$products = new WP_Query(
				array(
					'post_type' => 'product',
					'p'         => $product_id,
				)
			);

			while ( $products->have_posts() ) :
				$products->the_post();

				if ( post_password_required() ) {
					continue;
				}

				?>
				<button class="th-quickview__close" type="button">
					<?php
					// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
					echo apply_filters(
						'redparts_sputnik_quickview_close_button_content',
						esc_html__( 'Close', 'redparts-sputnik' )
					);
					?>
				</button>
				<div id="product-<?php the_ID(); ?>" <?php wc_product_class( 'th-product th-product--layout--quickview th-quickview__product' ); ?>>
					<?php
					/**
					 * Hook: redparts_sputnik_quickview_product.
					 *
					 * @hooked woocommerce_show_product_sale_flash - 100
					 */
					do_action( 'redparts_sputnik_quickview_product' );
					?>
					<div class="th-product__gallery th-product-gallery">
						<?php
						/**
						 * Hook: redparts_sputnik_quickview_gallery.
						 *
						 * @hooked woocommerce_show_product_images     - 100
						 */
						do_action( 'redparts_sputnik_quickview_gallery' );
						?>
					</div>
					<div class="th-product__summary">
						<?php
						/**
						 * Hook: redparts_sputnik_quickview_summary.
						 *
						 * @hooked woocommerce_template_single_title       - 100
						 * @hooked woocommerce_template_single_rating      - 200
						 * @hooked woocommerce_template_single_meta        - 300
						 * @hooked woocommerce_template_single_excerpt     - 400
						 * @hooked woocommerce_template_single_price       - 500
						 * @hooked woocommerce_template_single_add_to_cart - 600
						 */
						do_action( 'redparts_sputnik_quickview_summary' );
						?>
					</div>
				</div>
				<a class="th-quickview__see-details" href="<?php echo esc_url( get_permalink() ); ?>">
					<?php echo esc_html__( 'See full details', 'redparts-sputnik' ); ?>
				</a>
				<?php
			endwhile;

			wp_reset_postdata();
		}

		/**
		 * Returns quickview content.
		 *
		 * @param int $product_id Product id.
		 * @return string
		 */
		public function get_content( int $product_id ): string {
			ob_start();

			$this->the_content( $product_id );

			return ob_get_clean();
		}

		/**
		 * Handles AJAX requests to load a quickview content.
		 *
		 * @noinspection DuplicatedCode
		 */
		public function ajax() {
			if ( ! isset( $_GET['nonce'] ) || ! isset( $_GET['action'] ) || ! isset( $_GET['product_id'] ) ) {
				wp_send_json_error();
				wp_die();
			}

			WPML::switch_ajax_language();

			$nonce      = sanitize_key( wp_unslash( $_GET['nonce'] ) );
			$action     = sanitize_key( wp_unslash( $_GET['action'] ) );
			$product_id = absint( wp_unslash( $_GET['product_id'] ) );

			if ( ! wp_verify_nonce( $nonce, $action ) ) {
				wp_send_json_error();
				wp_die();
			}

			// With this WooCommerce will think that it is a product page.
			wp( 'p=' . $product_id . '&post_type=product' );

			wp_send_json(
				apply_filters(
					'redparts_sputnik_quickview_ajax_response',
					array(
						'success' => true,
						'content' => $this->get_content( $product_id ),
					),
					$action,
					$product_id
				)
			);
			wp_die();
		}

		/**
		 * Outputs sale badge.
		 */
		public function sale_badge() {
			ob_start();

			echo '<div class="th-product__badge">';

			ob_start();

			woocommerce_show_product_sale_flash();

			$has_sale_badge = ! empty( ob_get_contents() );

			ob_end_flush();

			echo '</div>';

			if ( $has_sale_badge ) {
				ob_end_flush();
			} else {
				ob_end_clean();
			}
		}

		/**
		 * Outputs stock badge.
		 *
		 * @noinspection DuplicatedCode
		 */
		public function stock_badge() {
			global $product;

			if ( ! $product->is_purchasable() ) {
				return;
			}

			$stock_html = wc_get_stock_html( $product );

			if ( $stock_html ) {
				echo '<div class="th-product__stock">';
				echo wp_kses( $stock_html, 'redparts_sputnik_stock' );
				echo '</div>';
			}
		}
	}
}
