<?php
/**
 * RedParts social links.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 *
 * @noinspection DuplicatedCode
 */

namespace RedParts\Sputnik;

use WP_Post;
use WP_Term;
use stdClass;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Social_Links' ) ) {
	/**
	 * Class Social_Links.
	 *
	 * @package RedParts\Sputnik
	 */
	class Social_Links extends Singleton {
		/**
		 * Array of available links.
		 *
		 * @var array
		 */
		private $available_links = array();

		/**
		 * Initialization.
		 */
		public function init() {
			$this->available_links = array(
				'facebook'   => array(
					'name'  => esc_html__( 'Facebook', 'redparts-sputnik' ),
					'icon'  => 'fa fa-facebook',
					'color' => '#4267b2',
				),
				'twitter'    => array(
					'name'  => esc_html__( 'Twitter', 'redparts-sputnik' ),
					'icon'  => 'fa fa-twitter',
					'color' => '#1b95e0',
				),
				'youtube'    => array(
					'name'  => esc_html__( 'Youtube', 'redparts-sputnik' ),
					'icon'  => 'fa fa-youtube-play',
					'color' => '#e52e2e',
				),
				'instagram'  => array(
					'name'  => esc_html__( 'Instagram', 'redparts-sputnik' ),
					'icon'  => 'fa fa-instagram',
					'color' => '#815dc7',
				),
				'rss'        => array(
					'name'  => esc_html__( 'RSS', 'redparts-sputnik' ),
					'icon'  => 'fa fa-rss',
					'color' => '#ffc338',
				),
				'pinterest'  => array(
					'name'  => esc_html__( 'Pinterest', 'redparts-sputnik' ),
					'icon'  => 'fa fa-pinterest',
					'color' => '#e81b22',
				),
				'tumblr'     => array(
					'name'  => esc_html__( 'Tumblr', 'redparts-sputnik' ),
					'icon'  => 'fa fa-tumblr',
					'color' => '#395976',
				),
				'linkedin'   => array(
					'name'  => esc_html__( 'LinkedIn', 'redparts-sputnik' ),
					'icon'  => 'fa fa-linkedin',
					'color' => '#006599',
				),
				'vimeo'      => array(
					'name'  => esc_html__( 'Vimeo', 'redparts-sputnik' ),
					'icon'  => 'fa fa-vimeo',
					'color' => '#1ab7ea',
				),
				'flickr'     => array(
					'name'  => esc_html__( 'Flickr', 'redparts-sputnik' ),
					'icon'  => 'fa fa-flickr',
					'color' => '#ff0084',
				),
				'github'     => array(
					'name'  => esc_html__( 'Github', 'redparts-sputnik' ),
					'icon'  => 'fa fa-github',
					'color' => '#1b1f23',
				),
				'dribbble'   => array(
					'name'  => esc_html__( 'Dribbble', 'redparts-sputnik' ),
					'icon'  => 'fa fa-dribbble',
					'color' => '#ea4c89',
				),
				'behance'    => array(
					'name'  => esc_html__( 'Behance', 'redparts-sputnik' ),
					'icon'  => 'fa fa-behance',
					'color' => '#0057ff',
				),
				'soundcloud' => array(
					'name'  => esc_html__( 'SoundCloud', 'redparts-sputnik' ),
					'icon'  => 'fa fa-soundcloud',
					'color' => '#ff5500',
				),
				'spotify'    => array(
					'name'  => esc_html__( 'Spotify', 'redparts-sputnik' ),
					'icon'  => 'fa fa-spotify',
					'color' => '#1ed760',
				),
				'ok'         => array(
					'name'  => esc_html__( 'Odnoklassniki', 'redparts-sputnik' ),
					'icon'  => 'fa fa-odnoklassniki',
					'color' => '#eb7815',
				),
				'vk'         => array(
					'name'  => esc_html__( 'Vkontakte', 'redparts-sputnik' ),
					'icon'  => 'fa fa-vk',
					'color' => '#4a76a8',
				),
				'whatsapp'   => array(
					'name'  => esc_html__( 'WhatsApp', 'redparts-sputnik' ),
					'icon'  => 'fa fa-whatsapp',
					'color' => '#09c03f',
				),
				'telegram'   => array(
					'name'  => esc_html__( 'Telegram', 'redparts-sputnik' ),
					'icon'  => 'fa fa-paper-plane',
					'color' => '#2ba0da',
				),
				'snapchat'   => array(
					'name'  => esc_html__( 'Snapchat', 'redparts-sputnik' ),
					'icon'  => 'fa fa-snapchat',
					'color' => '#f8f501',
				),
				'email'      => array(
					'name'  => esc_html__( 'Email', 'redparts-sputnik' ),
					'icon'  => 'fa fa-envelope',
					'color' => '#404040',
				),
			);

			add_action( 'admin_head-nav-menus.php', array( $this, 'init_nav_menu_meta_box' ) );
			add_filter( 'walker_nav_menu_start_el', array( $this, 'nav_menu_social_icons' ), 10, 4 );
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
			add_shortcode( 'redparts_sputnik_social_links', array( $this, 'shortcode' ) );
		}

		/**
		 * Adds a nav menu meta box.
		 *
		 * @noinspection PhpUnused
		 */
		public function init_nav_menu_meta_box() {
			add_meta_box(
				'redparts_sputnik_social_links',
				esc_html__( 'RedParts Social Links', 'redparts-sputnik' ),
				array( $this, 'nav_menu_meta_box' ),
				'nav-menus',
				'side',
				'high'
			);
		}

		/**
		 * Enqueue scripts.
		 */
		public function enqueue_scripts() {
			wp_add_inline_style( 'redparts-sputnik-style', $this->render_css() );
		}

		/**
		 * Returns available social links.
		 *
		 * @return array
		 */
		public function get_available_links(): array {
			return apply_filters( 'redparts_sputnik_get_available_social_links', $this->available_links );
		}

		/**
		 * Renders and prints social links HTML.
		 *
		 * @param string $classes CSS classes of the root HTML element.
		 */
		public function render( string $classes = '' ) {
			$menu_id = $this->get_menu_id();

			ob_start();

			$menu = wp_get_nav_menu_object( $menu_id );

			if ( $menu ) {
				wp_nav_menu(
					array(
						'container_class' => $classes . ' th-social-links',
						'menu'            => $menu,
						'depth'           => 1,
						'link_before'     => '<span class="sr-only">',
						'link_after'      => '</span>',
						'fallback_cb'     => '__return_empty_string',
					)
				);
			}

			// Because we have already escaped the output above.
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo apply_filters( 'redparts_sputnik_social_links_render', ob_get_clean(), $classes );
		}

		/**
		 * Renders and returns social links CSS.
		 *
		 * @return string
		 */
		public function render_css(): string {
			$links  = $this->get_available_links();
			$result = '';

			foreach ( $links as $key => $link ) {
				$result .= '.th-social-links .th-social-' . sanitize_key( $key ) . ' a';
				$result .= '{background: ' . sanitize_hex_color( $link['color'] ) . ';}';
			}

			return apply_filters( 'redparts_sputnik_social_links_render_css', $result );
		}

		/**
		 * Social links shortcode.
		 *
		 * @param string|array $attrs - Shortcode attributes.
		 *
		 * @return string
		 */
		public function shortcode( $attrs ): string {
			$attrs = shortcode_atts(
				array(
					'classes' => '',
				),
				$attrs
			);

			ob_start();

			$this->render( $attrs['classes'] );

			return ob_get_clean();
		}

		/**
		 * Adds icons to the nav menu items.
		 *
		 * @noinspection PhpUnusedParameterInspection,PhpMissingParamTypeInspection
		 *
		 * @param string   $item_output - The menu item output.
		 * @param WP_Post  $item        - Menu item object.
		 * @param int      $depth       - Depth of the menu.
		 * @param stdClass $args        - wp_nav_menu() arguments.
		 *
		 * @return string
		 */
		public function nav_menu_social_icons( string $item_output, $item, int $depth, stdClass $args ): string {
			static $added_styles = array();

			if ( ! isset( $args->menu ) ) {
				return $item_output;
			}

			$menu_id = $this->get_menu_id();

			if ( $menu_id === $args->menu || ( $args->menu instanceof WP_Term && $menu_id === $args->menu->term_id ) ) {
				$available_links = $this->get_available_links();
				/** Nav menu item has classes property. @noinspection PhpUndefinedFieldInspection */
				$classes        = explode( ' ', implode( ' ', $item->classes ) );
				$icon_classes   = array();
				$social_network = '';
				$color          = '';

				foreach ( $classes as $class ) {
					if ( preg_match( '#^th-icon-(.+)$#', $class, $mr ) ) {
						$icon_classes[] = $mr[1];
					}
					if ( preg_match( '#^th-social-(.+)$#', $class, $mr ) ) {
						$social_network = $mr[1];
					}
					if ( preg_match( '#^th-color-((?:[A-Fa-f0-9]{3}){1,2})$#', $class, $mr ) ) {
						$color = $mr[1];
					}
				}

				if ( $social_network && empty( $icon_classes ) && isset( $available_links[ $social_network ] ) ) {
					$icon_classes = explode( ' ', $available_links[ $social_network ]['icon'] );
				}

				$icon_classes = implode( ' ', $icon_classes );
				$icon_classes = $icon_classes ? $icon_classes : 'fa fa-link';
				$item_output  = str_replace( $args->link_after, '</span><i class="' . esc_attr( $icon_classes ) . '"></i>', $item_output );

				if ( $color ) {
					$style_id = $menu_id . '-' . $item->ID;

					if ( ! isset( $added_styles[ $style_id ] ) ) {
						$added_styles[ $style_id ] = true;

						$style = '.th-social-links .menu-item-' . $item->ID . ' a {background: #' . $color . '}';

						wp_add_inline_style( 'redparts-sputnik-inline', $style );
					}

					if ( is_customize_preview() ) {
						$item_output = str_replace( '<a', '<a style="background: ' . sanitize_hex_color( '#' . $color ) . '"', $item_output );
					}
				}
			}

			return $item_output;
		}

		/**
		 * Outputs nav menu meta box.
		 */
		public function nav_menu_meta_box() {
			$available_links = $this->get_available_links();

			?>
			<div id="post-type-redparts-social-links" class="posttypediv">
				<div id="tabs-panel-redparts-social-links" class="tabs-panel tabs-panel-active">
					<ul id="redparts-social-links-checklist" class="categorychecklist form-no-clear">
						<?php
						$i = -1;
						foreach ( $available_links as $key => $value ) :
							?>
							<li>
								<label class="menu-item-title">
									<input
										type="checkbox"
										class="menu-item-checkbox"
										name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-object-id]"
										value="<?php echo esc_attr( $i ); ?>"
									/> <?php echo esc_html( $value['name'] ); ?>
								</label>
								<input
									type="hidden"
									class="menu-item-type"
									name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-type]"
									value="custom"
								/>
								<input
									type="hidden"
									class="menu-item-title"
									name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-title]"
									value="<?php echo esc_attr( $value['name'] ); ?>"
								/>
								<input
									type="hidden"
									class="menu-item-url"
									name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-url]"
								/>
								<input
									type="hidden"
									class="menu-item-classes"
									name="menu-item[<?php echo esc_attr( $i ); ?>][menu-item-classes]"
									value="<?php echo esc_attr( 'th-social-' . $key ); ?>"
								/>
							</li>
							<?php
							$i--;
						endforeach;
						?>
					</ul>
				</div>
				<p class="button-controls">
					<span class="list-controls">
						<a
							href="<?php echo esc_url( admin_url( 'nav-menus.php?page-tab=all&selectall=1#post-type-redparts-social-links' ) ); ?>"
							class="select-all"
						>
							<?php esc_html_e( 'Select all', 'redparts-sputnik' ); ?>
						</a>
					</span>
					<span class="add-to-menu">
						<button
							type="submit"
							class="button-secondary submit-add-to-menu right"
							value="<?php esc_attr_e( 'Add to menu', 'redparts-sputnik' ); ?>"
							name="add-post-type-menu-item"
							id="submit-post-type-redparts-social-links"
						>
							<?php esc_html_e( 'Add to menu', 'redparts-sputnik' ); ?>
						</button>
						<span class="spinner"></span>
					</span>
				</p>
			</div>
			<?php
		}

		/**
		 * Returns ID of social links menu.
		 *
		 * @since 1.6.0
		 *
		 * @return int
		 */
		public function get_menu_id(): int {
			$menu_id = absint( Settings::instance()->get( 'social_links_menu' ) );

			return absint( apply_filters( 'wpml_object_id', $menu_id, 'nav_menu', true ) );
		}
	}
}
