<?php
/**
 * Base Widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik;

use Elementor\Plugin;
use WP_Widget;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widget' ) ) {
	/**
	 * Class Widget
	 */
	class Widget extends WP_Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = true;

		/**
		 * Indicates that the widget is already registered or not.
		 *
		 * @var bool
		 */
		protected $registered = false;

		/**
		 * Echoes the widget content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		public function widget( $args, $instance ) {
			echo $args['before_widget']; // phpcs:ignore

			if ( $this->display_title ) {
				$title = $this->get_title( $instance );

				if ( $title ) {
					echo $args['before_title']; // phpcs:ignore
					echo apply_filters( 'widget_title', $title, $instance, $this->id_base ); // phpcs:ignore
					echo $args['after_title']; // phpcs:ignore
				}
			}

			$fields = $this->form_fields();
			$fields = $fields ? $fields : array();

			$this->widget_body( (array) $args, $this->translate( $fields, $instance ) );

			echo $args['after_widget']; // phpcs:ignore
		}

		/**
		 * Returns widget title.
		 *
		 * @param array $instance The settings for the particular instance of the widget.
		 *
		 * @return string
		 */
		public function get_title( array $instance ): string {
			$title = ! empty( $instance['title'] ) ? $instance['title'] : '';
			$title = apply_filters( 'widget_title', $title, $instance, $this->id_base );

			return $title;
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) { }

		/**
		 * Checks if the widget works in the context of the elementor editor.
		 *
		 * @return bool
		 */
		public function is_elementor_editor(): bool {
			if ( ! class_exists( '\Elementor\Plugin' ) ) {
				return false;
			}

			return Plugin::$instance->editor->is_edit_mode();
		}

		/**
		 * Checks if the current widget is being used in the Elementor context.
		 *
		 * @since 1.6.0
		 *
		 * @return bool
		 */
		public function is_elementor_widget(): bool {
			/**
			 * $this->number can have a string type.
			 *
			 * @noinspection PhpStrictComparisonWithOperandsOfDifferentTypesInspection
			 */
			return 'REPLACE_TO_ID' === $this->number;
		}

		/**
		 * Returns form fields.
		 *
		 * @return array|false
		 */
		public function form_fields() {
			return false;
		}

		/**
		 * Returns translated settings.
		 *
		 * @since 1.6.0
		 *
		 * @param array    $fields   Array with declaration of form fields.
		 * @param array    $instance Widget settings.
		 * @param string[] $path     Path to the fields.
		 */
		public function translate( array $fields, array $instance, array $path = array() ): array {
			foreach ( $fields as $field ) {
				$field_key  = $field['key'];
				$field_type = $field['type'];

				if ( ! isset( $instance[ $field_key ] ) ) {
					continue;
				}

				switch ( $field_type ) {
					case 'text':
					case 'textarea':
						if ( ! empty( $field['translatable'] ) && $field['translatable'] && ! $this->is_elementor_widget() ) {
							$string_name = $this->wpml_string_name( array_merge( $path, array( $field_key ) ) );

							$instance[ $field_key ] = apply_filters( 'wpml_translate_single_string', $instance[ $field_key ], 'Widgets', $string_name );
						}

						break;
					case 'tabs':
						if ( ! is_array( $instance[ $field_key ] ) ) {
							break;
						}

						foreach ( $instance[ $field_key ] as $tab_idx => $tab_instance ) {
							$instance[ $field_key ][ $tab_idx ] = $this->translate(
								$field['fields'],
								$tab_instance,
								array_merge(
									$path,
									array(
										$field_key,
										$tab_idx + 1,
									)
								)
							);
						}

						break;
				}
			}

			return $instance;
		}

		/**
		 * Updates a particular instance of a widget.
		 *
		 * @param array $new_instance New settings for this instance as input by the user via WP_Widget::form().
		 * @param array $old_instance Old settings for this instance.
		 *
		 * @return array Settings to save or bool false to cancel saving.
		 */
		public function update( $new_instance, $old_instance ): array {
			$fields = $this->form_fields();

			if ( ! $fields ) {
				return $new_instance;
			}

			return $this->update_fields( $fields, $new_instance );
		}

		/**
		 * Updates a particular instance of a widget.
		 *
		 * @param array    $fields       Array with declaration of form fields.
		 * @param array    $new_instance New settings for this instance as input by the user via WP_Widget::form().
		 * @param string[] $path         Path to the fields.
		 *
		 * @return array Settings to save.
		 */
		public function update_fields( array $fields, array $new_instance, array $path = array() ): array {
			$instance = array();

			foreach ( $fields as $field ) {
				$field_key  = $field['key'];
				$field_type = $field['type'];

				if ( ! isset( $new_instance[ $field_key ] ) ) {
					$instance[ $field_key ] = $field['default'] ?? '';

					continue;
				}

				switch ( $field_type ) {
					case 'text':
					case 'textarea':
						if ( current_user_can( 'unfiltered_html' ) ) {
							$instance[ $field_key ] = $new_instance[ $field_key ];
						} else {
							$instance[ $field_key ] = wp_kses_post( $new_instance[ $field_key ] );
						}

						if ( ! empty( $field['translatable'] ) && $field['translatable'] && ! $this->is_elementor_widget() ) {
							$string_name = $this->wpml_string_name( array_merge( $path, array( $field_key ) ) );

							do_action( 'wpml_register_single_string', 'Widgets', $string_name, $instance[ $field_key ] );
						}

						break;
					case 'number':
						$instance[ $field_key ] = intval( $new_instance[ $field_key ] );

						break;
					case 'select':
						$valid_values = array_map(
							function ( $option ) {
								return $option['key'];
							},
							$field['options']
						);

						if ( in_array( $new_instance[ $field_key ], $valid_values, true ) ) {
							$instance[ $field_key ] = $new_instance[ $field_key ];
						}

						break;
					case 'image':
						$instance[ $field_key ] = absint( $new_instance[ $field_key ] );

						break;
					case 'tabs':
						if ( isset( $field['select_key'] ) && isset( $new_instance[ $field['select_key'] ] ) ) {
							$indexes       = array_map( 'intval', array_keys( $new_instance[ $field_key ] ) );
							$current_index = intval( $new_instance[ $field['select_key'] ] );

							$instance[ $field['select_key'] ] = array_search( $current_index, $indexes, true );
						}

						$instance[ $field_key ] = array();

						if ( isset( $new_instance[ $field_key ] ) ) {
							foreach ( $new_instance[ $field_key ] as $item_new_instance ) {
								$instance[ $field_key ][] = $this->update_fields(
									$field['fields'],
									$item_new_instance,
									array_merge(
										$path,
										array(
											$field_key,
											count( $instance[ $field_key ] ) + 1,
										)
									)
								);
							}
						}

						break;
					default:
						$instance[ $field_key ] = sanitize_text_field( $new_instance[ $field_key ] );
				}
			}

			return $instance;
		}

		/**
		 * Outputs the settings update form.
		 *
		 * @param  array $instance Current settings.
		 *
		 * @return string Default return is 'noform'.
		 */
		public function form( $instance ): string {
			$fields = $this->form_fields();

			if ( empty( $fields ) ) {
				return 'noform';
			}

			$this->render_form( $fields, $instance );

			return '';
		}

		/**
		 * Render widget form.
		 *
		 * @param array  $fields   Form fields declaration.
		 * @param array  $instance Current settings.
		 * @param string $context  Fields context.
		 */
		protected function render_form( array $fields, array $instance, $context = '' ) {
			foreach ( $fields as $field ) :
				$path = '' === $context ? $field['key'] : $context . '[' . $field['key'] . ']';

				?>
				<div class="redparts-form-row">
					<label for="<?php echo esc_attr( $this->get_field_id( $path ) ); ?>">
						<?php echo esc_html( $field['label'] ); ?>
					</label>
					<?php if ( 'text' === $field['type'] ) : ?>
						<?php $value = isset( $instance[ $field['key'] ] ) ? $instance[ $field['key'] ] : ''; ?>
						<input
							class="widefat"
							type="text"
							id="<?php echo esc_attr( $this->get_field_id( $path ) ); ?>"
							name="<?php echo esc_attr( $this->get_field_name( $path ) ); ?>"
							value="<?php echo esc_attr( $value ); ?>"
							<?php if ( ! empty( $field['placeholder'] ) ) : ?>
								placeholder="<?php echo esc_attr( $field['placeholder'] ); ?>"
							<?php endif; ?>
						/>
					<?php elseif ( 'color' === $field['type'] ) : ?>
						<?php $value = isset( $instance[ $field['key'] ] ) ? $instance[ $field['key'] ] : ''; ?>
						<input
							class="widefat redparts-color-picker"
							type="text"
							id="<?php echo esc_attr( $this->get_field_id( $path ) ); ?>"
							name="<?php echo esc_attr( $this->get_field_name( $path ) ); ?>"
							value="<?php echo esc_attr( $value ); ?>"/>
					<?php elseif ( 'number' === $field['type'] ) : ?>
						<?php $value = $instance[ $field['key'] ] ?? ( $field['default'] ?? '' ); ?>
						<input
							class="widefat"
							type="number"
							id="<?php echo esc_attr( $this->get_field_id( $path ) ); ?>"
							name="<?php echo esc_attr( $this->get_field_name( $path ) ); ?>"
							value="<?php echo esc_attr( $value ); ?>"/>
					<?php elseif ( 'textarea' === $field['type'] ) : ?>
						<?php $value = isset( $instance[ $field['key'] ] ) ? $instance[ $field['key'] ] : ''; ?>
						<textarea
							class="widefat"
							id="<?php echo esc_attr( $this->get_field_id( $path ) ); ?>"
							name="<?php echo esc_attr( $this->get_field_name( $path ) ); ?>"><?php echo esc_html( $value ); ?></textarea>
					<?php elseif ( 'select' === $field['type'] ) : ?>
						<?php $value = $instance[ $field['key'] ] ?? ( $field['default'] ?? '' ); ?>
						<select
							class="widefat"
							id="<?php echo esc_attr( $this->get_field_id( $path ) ); ?>"
							name="<?php echo esc_attr( $this->get_field_name( $path ) ); ?>">
							<?php foreach ( $field['options'] as $option ) : ?>
								<option value="<?php echo esc_attr( $option['key'] ); ?>"<?php selected( $option['key'], $value ); ?>>
									<?php echo esc_html( $option['label'] ); ?>
								</option>
							<?php endforeach; ?>
						</select>
					<?php elseif ( 'image' === $field['type'] ) : ?>
						<?php
						$value   = isset( $instance[ $field['key'] ] ) ? absint( $instance[ $field['key'] ] ) : '';
						$image   = wp_get_attachment_image( $value, 'full' );
						$classes = array( 'redparts-image-picker' );

						if ( $image ) {
							$classes[] = 'redparts-image-picker--has-image';
						}

						?>
						<div class="<?php echo esc_attr( implode( ' ', $classes ) ); ?>">
							<input
								type="hidden"
								name="<?php echo esc_attr( $this->get_field_name( $path ) ); ?>"
								value="<?php echo esc_attr( $value ); ?>"
							>
							<button class="redparts-image-picker__button"
									type="button"><?php esc_html_e( 'Select Image', 'redparts-sputnik' ); ?></button>
							<div class="redparts-image-picker__container">
								<?php echo wp_kses( $image, 'redparts_sputnik_image' ); ?>
							</div>
							<div class="redparts-image-picker__actions">
								<a class="redparts-image-picker__remove redparts-link--warning" href="#">
									<?php esc_html_e( 'Remove Image', 'redparts-sputnik' ); ?>
								</a>
								<span>|</span>
								<a class="redparts-image-picker__replace" href="#">
									<?php esc_html_e( 'Replace Image', 'redparts-sputnik' ); ?>
								</a>
							</div>
						</div>
					<?php elseif ( 'tabs' === $field['type'] ) : ?>
						<?php $this->render_tabs( $field, $instance, $context ); ?>
					<?php endif; ?>
				</div>
				<?php
			endforeach;
		}

		/**
		 * Renders tabs.
		 *
		 * @param array  $field    Form field declaration.
		 * @param array  $instance Current settings.
		 * @param string $context  Field context.
		 */
		protected function render_tabs( array $field, array $instance, $context = '' ) {
			$value        = isset( $instance[ $field['key'] ] ) && is_array( $instance[ $field['key'] ] ) ? $instance[ $field['key'] ] : array();
			$path         = '' === $context ? $field['key'] : $context . '[' . $field['key'] . ']';
			$max          = isset( $field['max'] ) ? $field['max'] : 0;
			$classes      = array( 'redparts-tabs' );
			$select_path  = '';
			$select_value = 0;

			if ( isset( $field['select_key'] ) ) {
				$select_path  = '' === $context ? $field['select_key'] : $context . '[' . $field['select_key'] . ']';
				$select_value = isset( $instance[ $field['select_key'] ] ) ? intval( $instance[ $field['select_key'] ] ) : $select_value;
			}

			if ( 0 < $max && $max <= count( $value ) ) {
				$classes[] = 'redparts-tabs--max';
			}

			$active_panel_index = array_search( $select_value, array_keys( $value ), true );
			$active_panel_index = $active_panel_index ? $active_panel_index : -1;

			// phpcs:disable Generic.PHP.DisallowAlternativePHPTags.MaybeASPShortOpenTagFound
			// phpcs:disable Generic.PHP.DisallowAlternativePHPTags.MaybeASPOpenTagFound
			?>
			<div
				class="<?php echo esc_attr( implode( ' ', $classes ) ); ?>"
				data-max="<?php echo esc_attr( $max ); ?>"
				data-panel-title="<?php echo esc_attr( $field['l10n']['title'] ); ?>"
			>
				<a class="redparts-tabs__add" href="#"><?php echo esc_html( $field['l10n']['add'] ); ?></a>
				<select
					class="redparts-tabs__list"
					<?php if ( $select_path ) : ?>
						name="<?php echo esc_attr( $this->get_field_name( $select_path ) ); ?>"
					<?php endif; ?>
					id="<?php echo esc_attr( $this->get_field_id( $path ) ); ?>"
				>
					<?php foreach ( $value as $tab_index => $tab_instance ) : ?>
						<?php
						$tab_title = sprintf( $field['l10n']['title'], $tab_index + 1 );

						if ( isset( $field['title'] ) && isset( $tab_instance[ $field['title'] ] ) ) {
							$tab_title = $tab_instance[ $field['title'] ];
						}
						?>
						<option
							value="<?php echo esc_attr( $tab_index ); ?>"
							<?php if ( $tab_index === $select_value ) : ?>
								selected
							<?php endif; ?>
						>
							<?php echo esc_html( $tab_title ); ?>
						</option>
					<?php endforeach; ?>
				</select>
				<?php $first_tab = true; ?>
				<?php foreach ( $value as $tab_index => $tab_instance ) : ?>
					<?php $tab_context = $context . $field['key'] . '[' . $tab_index . ']'; ?>
					<div
						class="
					redparts-tabs__panel
					<?php if ( $tab_index === $active_panel_index || ( -1 === $active_panel_index && $first_tab ) ) : ?>
						redparts-tabs__panel--active
					<?php endif; ?>
					"
						data-index="<?php echo esc_attr( $tab_index ); ?>"
					>
						<div class="redparts-tabs__panel-content">
							<?php $this->render_form( $field['fields'], $tab_instance, $tab_context ); ?>
						</div>
						<div class="redparts-tabs__panel-footer">
							<a class="redparts-tabs__remove" href="#"><?php echo esc_html( $field['l10n']['remove'] ); ?></a>
						</div>
					</div>
					<?php $first_tab = false; ?>
				<?php endforeach; ?>
				<script type="text/html" class="redparts-tabs__panel-template">
					<% context_id   = <?php echo wp_json_encode( $this->get_field_id( $context . $field['key'] ) ); ?> + '-' + index %>
					<% context_name = <?php echo wp_json_encode( $this->get_field_name( $context . $field['key'] ) ); ?> + '[' + index + ']' %>
					<div class="redparts-tabs__panel" data-index="<%= index %>">
						<div class="redparts-tabs__panel-content">
							<?php $this->render_form_template( $field['fields'] ); ?>
						</div>
						<div class="redparts-tabs__panel-footer">
							<a class="red-arts-tabs__remove" href="#"><?php echo esc_html( $field['l10n']['remove'] ); ?></a>
						</div>
					</div>
				</script>
			</div>
			<?php
			// phpcs:enable
		}

		/**
		 * Render the form template that will be used in JavaScript.
		 *
		 * @param array $fields Form fields.
		 */
		public function render_form_template( array $fields ) {
			// phpcs:disable Generic.PHP.DisallowAlternativePHPTags.MaybeASPShortOpenTagFound
			// phpcs:disable Generic.PHP.DisallowAlternativePHPTags.MaybeASPOpenTagFound
			foreach ( $fields as $field ) :
				?>
				<% field_id = context_id + '-' + <?php echo wp_json_encode( $field['key'] ); ?> %>
				<% field_name = context_name + '[' + <?php echo wp_json_encode( $field['key'] ); ?> + ']' %>
				<div class="redparts-form-row">
					<label for="<%= field_id %>">
						<?php echo esc_html( $field['label'] ); ?>
					</label>
					<?php if ( 'text' === $field['type'] ) : ?>
						<input class="widefat" type="text" id="<%= field_id %>" name="<%= field_name %>" value="">
					<?php elseif ( 'color' === $field['type'] ) : ?>
						<input class="widefat redparts-color-picker" type="text" id="<%= field_id %>" name="<%= field_name %>" value="">
					<?php elseif ( 'number' === $field['type'] ) : ?>
						<input class="widefat" type="number" id="<%= field_id %>" name="<%= field_name %>" value="">
					<?php elseif ( 'textarea' === $field['type'] ) : ?>
						<textarea
							class="widefat"
							id="<%= field_id %>"
							name="<%= field_name %>"
						></textarea>
					<?php elseif ( 'select' === $field['type'] ) : ?>
						<select class="widefat" id="<%= field_id %>" name="<%= field_name %>">
							<?php foreach ( $field['options'] as $option ) : ?>
								<option value="<?php echo esc_attr( $option['key'] ); ?>">
									<?php echo esc_html( $option['label'] ); ?>
								</option>
							<?php endforeach; ?>
						</select>
					<?php elseif ( 'image' === $field['type'] ) : ?>
						<div class="redparts-image-picker">
							<input type="hidden" name="<%= field_name %>" value="">
							<button class="redparts-image-picker__button"
									type="button"><?php esc_html_e( 'Select Image', 'redparts-sputnik' ); ?></button>
							<div class="redparts-image-picker__container"></div>
							<div class="redparts-image-picker__actions">
								<a class="redparts-image-picker__remove redparts-link--warning" href="#">
									<?php esc_html_e( 'Remove Image', 'redparts-sputnik' ); ?>
								</a>
								<span>|</span>
								<a class="redparts-image-picker__replace" href="#">
									<?php esc_html_e( 'Replace Image', 'redparts-sputnik' ); ?>
								</a>
							</div>
						</div>
					<?php endif; ?>
				</div>
				<?php
			endforeach;
			// phpcs:enable
		}

		/**
		 * Add hooks while registering all widget instances of this widget class.
		 *
		 * @param integer $number Optional. The unique order number of this widget instance
		 *                        compared to other instances of the same class. Default -1.
		 */
		public function _register_one( $number = -1 ) { // phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore
			parent::_register_one( $number );

			if ( $this->registered ) {
				return;
			}

			$this->registered = true;
			$this->register_hooks();
		}

		/**
		 * Register hooks.
		 */
		public function register_hooks() {
			// Note that the widgets component in the customizer will also do the 'admin_print_scripts-widgets.php'
			// action in WP_Customize_Widgets::print_scripts().
			add_action( 'admin_print_scripts-widgets.php', array( $this, 'enqueue_admin_scripts' ) );

			if ( $this->is_preview() ) {
				add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_preview_scripts' ) );
			}

			if ( ! is_admin() ) {
				add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_scripts' ) );
			}
		}

		/**
		 * Enqueue admin scripts.
		 */
		public function enqueue_admin_scripts() {
		}

		/**
		 * Enqueue preview scripts.
		 *
		 * @noinspection PhpUnused
		 */
		public function enqueue_preview_scripts() {
		}

		/**
		 * Enqueue frontend scripts.
		 */
		public function enqueue_frontend_scripts() {
		}

		/**
		 * Returns the string name of the field for WPML.
		 *
		 * @since 1.6.0
		 *
		 * @param array $path Field path.
		 * @return string
		 */
		protected function wpml_string_name( array $path ): string {
			$pre_path = array( $this->id_base );

			if ( ! $this->is_elementor_widget() ) {
				$pre_path[] = $this->number;
			}

			$path  = array_merge( $pre_path, $path );
			$field = array_pop( $path );

			return '[' . implode( '][', $path ) . ']' . $field;
		}
	}
}
