<?php
/**
 * This file contains code related to the attribute swatches.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Swatches;

use RedParts\Sputnik\Singleton;
use WC_Product_Variable;
use WC_Product_Attribute;
use stdClass;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Swatches\Swatches' ) ) {
	/**
	 * Class Swatches
	 */
	class Swatches extends Singleton {
		/**
		 * Initialization.
		 */
		public function init() {
			add_filter( 'product_attributes_type_selector', array( $this, 'get_attribute_types' ) );
			add_filter( 'woocommerce_product_option_terms', array( $this, 'product_terms_selection_field' ), 10, 3 );
			add_filter( 'woocommerce_dropdown_variation_attribute_options_html', array( $this, 'control_template' ), 10, 2 );
		}

		/**
		 * Returns an associative array of attribute types.
		 *
		 * @param array $types An associative array of attribute types.
		 * @return array
		 */
		public function get_attribute_types( $types = array() ): array {
			return array_merge( $types, apply_filters( 'redparts_sputnik_swatches_types', array() ) );
		}

		/**
		 * Outputs terms selection field.
		 *
		 * @param stdClass             $attribute_taxonomy Attribute taxonomy.
		 * @param int                  $i                  Attribute index.
		 * @param WC_Product_Attribute $attribute          Attribute object.
		 */
		public function product_terms_selection_field( stdClass $attribute_taxonomy, int $i, WC_Product_Attribute $attribute ) {
			$types = array_keys( $this->get_attribute_types() );

			if ( ! in_array( $attribute_taxonomy->attribute_type, $types, true ) ) {
				return;
			}

			?>
			<!--suppress HtmlFormInputWithoutLabel -->
			<select
				multiple="multiple"
				data-placeholder="<?php esc_attr_e( 'Select terms', 'redparts-sputnik' ); ?>"
				class="multiselect attribute_values wc-enhanced-select"
				name="attribute_values[<?php echo esc_attr( $i ); ?>][]"
			>
				<?php
				$args      = array(
					'orderby'    => ! empty( $attribute_taxonomy->attribute_orderby ) ? $attribute_taxonomy->attribute_orderby : 'name',
					'hide_empty' => 0,
				);
				$all_terms = get_terms( $attribute->get_taxonomy(), apply_filters( 'woocommerce_product_attribute_terms', $args ) );

				if ( $all_terms ) {
					foreach ( $all_terms as $term ) {
						$options = $attribute->get_options();
						$options = ! empty( $options ) ? $options : array();

						?>
						<option
							value="<?php echo esc_attr( $term->term_id ); ?>"
							<?php
							// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Already escaped in wc_selected.
							echo wc_selected( $term->term_id, $options );
							?>
						>
							<?php echo esc_attr( apply_filters( 'woocommerce_product_attribute_term_name', $term->name, $term ) ); ?>
						</option>
						<?php
					}
				}
				?>
			</select>
			<button class="button plus select_all_attributes"><?php esc_html_e( 'Select all', 'redparts-sputnik' ); ?></button>
			<button class="button minus select_no_attributes"><?php esc_html_e( 'Select none', 'redparts-sputnik' ); ?></button>
			<button class="button fr plus add_new_attribute"><?php esc_html_e( 'Add new', 'redparts-sputnik' ); ?></button>
			<?php
		}

		/**
		 * Outputs swatches control template.
		 *
		 * @param string $html Control template.
		 * @param array  $args Args array.
		 *
		 * @return string
		 */
		public function control_template( string $html, array $args ): string {
			/** Product object. @var WC_Product_Variable $product */
			/** Attribute object. @var WC_Product_Attribute $attribute */
			/** Attribute taxonomy. @var stdClass $taxonomy */

			$types      = $this->get_attribute_types();
			$product    = $args['product'];
			$options    = $args['options'];
			$attributes = $product->get_attributes();

			if ( ! isset( $attributes[ $args['attribute'] ] ) ) {
				return $html;
			}

			$attribute = $attributes[ $args['attribute'] ];
			$taxonomy  = $attribute->get_taxonomy_object();

			if ( ! $taxonomy || ! isset( $types[ $taxonomy->attribute_type ] ) ) {
				return $html;
			}

			$values = wc_get_product_terms( $product->get_id(), $args['attribute'], array( 'fields' => 'all' ) );
			$values = array_filter(
				$values,
				function( $value ) use ( $options ) {
					return in_array( $value->slug, $options, true );
				}
			);

			ob_start();

			do_action( 'redparts_sputnik_swatches_control_template', $args, $taxonomy->attribute_type, $values );

			return ob_get_clean() . $html;
		}
	}
}
