<?php
/**
 * Block Slideshow Widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Block_Slideshow' ) ) {
	/**
	 * Class Block_Slideshow
	 */
	class Block_Slideshow extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = false;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_options = array(
				'classname'                   => 'widget_redparts_sputnik_block_slideshow',
				'description'                 => esc_html_x( 'Shows a slide show.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_block_slideshow',
				esc_html_x( 'RedParts: Block Slideshow', 'Admin', 'redparts-sputnik' ),
				$widget_options
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 *
		 * @since 2.8.0
		 */
		protected function widget_body( array $args, array $instance ) {
			$autoplay             = ( $instance['autoplay'] ?? 'no' ) === 'yes';
			$autoplay_hover_pause = ( $instance['autoplay_hover_pause'] ?? 'yes' ) === 'yes';
			$autoplay_timeout     = absint( $instance['autoplay_timeout'] ?? '5000' );

			$slides = array();

			if ( isset( $instance['slides'] ) && is_array( $instance['slides'] ) ) {
				$slides = $instance['slides'];
			}

			uasort(
				$slides,
				function ( $a, $b ) {
					$a = intval( $a['order'] );
					$b = intval( $b['order'] );

					if ( $a === $b ) {
						return 0;
					}

					return ( $a < $b ) ? -1 : 1;
				}
			);

			$start_position = array_search( intval( $instance['current_slide'] ), array_keys( $slides ), true );
			$start_position = $start_position && ( $this->is_preview() || $this->is_elementor_editor() ) ? $start_position : 0;

			?>
			<!-- .th-block-slideshow -->
			<div
				class="th-block th-block-slideshow"
				data-start-position="<?php echo esc_attr( $start_position ); ?>"
				data-autoplay="<?php echo esc_attr( $autoplay ? 1 : 0 ); ?>"
				data-autoplay-hover-pause="<?php echo esc_attr( $autoplay_hover_pause ? 1 : 0 ); ?>"
				data-autoplay-timeout="<?php echo esc_attr( $autoplay_timeout ); ?>"
			>
				<div class="th-container">
					<?php $this->slides( $slides ); ?>
				</div>
			</div>
			<!-- .th-block-slideshow / end -->
			<?php
		}

		/**
		 * Echoes the slides.
		 *
		 * @param array $slides Slides array.
		 */
		protected function slides( array $slides ) {
			?>
			<div class="th-block-slideshow__carousel">
				<div class="owl-carousel">
					<?php foreach ( $slides as $slide ) : ?>
						<?php
						$badge       = $slide['badge'] ?? '';
						$title       = $slide['title'] ?? '';
						$subtitle    = $slide['subtitle'] ?? '';
						$button_text = $slide['button_text'] ?? '';
						$url         = $slide['url'] ?? '';

						$image_desktop = isset( $slide['image_desktop'] ) ? absint( $slide['image_desktop'] ) : '';
						$image_mobile  = isset( $slide['image_mobile'] ) ? absint( $slide['image_mobile'] ) : '';

						?>
						<a class="th-block-slideshow__item" href="<?php echo esc_url( $url ); ?>">
							<?php echo wp_get_attachment_image( $image_desktop, 'full', false, array( 'class' => 'th-block-slideshow__item-image th-block-slideshow__item-image--desktop' ) ); ?>
							<?php echo wp_get_attachment_image( $image_mobile, 'full', false, array( 'class' => 'th-block-slideshow__item-image th-block-slideshow__item-image--mobile' ) ); ?>

							<?php if ( $badge ) : ?>
								<span class="th-block-slideshow__item-offer"><?php echo esc_html( $badge ); ?></span>
							<?php endif; ?>

							<?php if ( $title ) : ?>
								<div class="th-block-slideshow__item-title"><?php echo wp_kses( $title, 'redparts_sputnik_block_slideshow_title' ); ?></div>
							<?php endif; ?>
							<?php if ( $subtitle ) : ?>
								<div class="th-block-slideshow__item-details"><?php echo wp_kses( $subtitle, 'redparts_sputnik_block_slideshow_details' ); ?></div>
							<?php endif; ?>
							<?php if ( $button_text ) : ?>
								<div class="th-block-slideshow__item-button">
									<?php echo esc_html( $button_text ); ?>
								</div>
							<?php endif; ?>
						</a>
					<?php endforeach; ?>
				</div>
			</div>
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'     => 'autoplay',
					'label'   => esc_html__( 'Autoplay', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'no',
				),
				array(
					'key'     => 'autoplay_hover_pause',
					'label'   => esc_html__( 'Pause autoplay on hover', 'redparts-sputnik' ),
					'type'    => 'select',
					'options' => array(
						array(
							'key'   => 'yes',
							'label' => esc_html__( 'Yes', 'redparts-sputnik' ),
						),
						array(
							'key'   => 'no',
							'label' => esc_html__( 'No', 'redparts-sputnik' ),
						),
					),
					'default' => 'yes',
				),
				array(
					'key'     => 'autoplay_timeout',
					'label'   => esc_html__( 'Autoplay timeout', 'redparts-sputnik' ),
					'type'    => 'number',
					'default' => '5000',
				),
				array(
					'key'        => 'slides',
					'select_key' => 'current_slide',
					'label'      => esc_html__( 'Slides', 'redparts-sputnik' ),
					'type'       => 'tabs',
					'fields'     => array(
						array(
							'key'   => 'order',
							'label' => esc_html__( 'Order', 'redparts-sputnik' ),
							'type'  => 'number',
						),
						array(
							'key'          => 'badge',
							'label'        => esc_html__( 'Badge', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'title',
							'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'subtitle',
							'label'        => esc_html__( 'Subtitle', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'button_text',
							'label'        => esc_html__( 'Button Text', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'          => 'url',
							'label'        => esc_html__( 'URL', 'redparts-sputnik' ),
							'type'         => 'text',
							'translatable' => true,
						),
						array(
							'key'   => 'image_desktop',
							'label' => esc_html__( 'Desktop Image', 'redparts-sputnik' ),
							'type'  => 'image',
						),
						array(
							'key'   => 'image_mobile',
							'label' => esc_html__( 'Mobile Image', 'redparts-sputnik' ),
							'type'  => 'image',
						),
					),
					'l10n'       => array(
						'add'    => esc_html__( 'Add Slide', 'redparts-sputnik' ),
						'remove' => esc_html__( 'Remove Slide', 'redparts-sputnik' ),
						/* translators: %s: Slide number. */
						'title'  => esc_html__( 'Slide %s', 'redparts-sputnik' ),
					),
				),
			);
		}

		/**
		 * Loads the required media files for the media manager and scripts for media widgets.
		 */
		public function enqueue_admin_scripts() {
			wp_enqueue_media();
			wp_enqueue_style( 'redparts-sputnik-admin' );
			wp_enqueue_script( 'redparts-sputnik-admin' );
		}

		/**
		 * Loads the required media files for the media manager and scripts for media widgets.
		 */
		public function enqueue_frontend_scripts() {
			wp_enqueue_script( 'redparts-sputnik-widget-block-slideshow' );
		}
	}
}
