<?php
/**
 * Filter vehicle widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Garage;
use RedParts\Sputnik\Settings;
use RedParts\Sputnik\Vehicles;
use RedParts\Sputnik\Widget;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Filter_Vehicle' ) ) {

	/**
	 * Class Filter_Vehicle
	 */
	class Filter_Vehicle extends Widget {
		/**
		 * Indicates whether to display the widget title or not.
		 *
		 * @var bool
		 */
		protected $display_title = true;

		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_filter_vehicle',
				'description'                 => esc_html_x( 'Displays vehicle filter.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_filter_vehicle',
				esc_html_x( 'RedParts: Vehicle filter', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 *
		 * @noinspection PhpMissingParamTypeInspection
		 */
		public function widget( $args, $instance ) {
			if ( 'yes' !== Settings::instance()->get( 'autoparts_features' ) ) {
				return;
			}

			parent::widget( $args, $instance );
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$vehicle             = Garage::instance()->get_current_vehicle( 'shop' );
			$vehicle_instance_id = $vehicle ? $vehicle['instance_id'] : '';

			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$checked = $vehicle && isset( $_GET[ Vehicles::instance()->get_attribute_filter_name() ] );

			?>
			<div class="th-filter-vehicle" data-th-current-vehicle-key="<?php echo esc_attr( $vehicle_instance_id ); ?>">
				<?php if ( empty( $vehicle ) ) : ?>
					<div class="th-filter-vehicle__empty">
						<?php echo esc_html__( 'Select a vehicle to filter out only exact fit parts.', 'redparts-sputnik' ); ?>
					</div>
				<?php else : ?>
					<ul class="th-filter-vehicle__list">
						<li class="th-filter-vehicle__item">
							<label class="th-filter-vehicle__item-label">
								<input
									class="th-filter-vehicle__item-input"
									type="checkbox"
									value="<?php echo esc_attr( Vehicles::instance()->get_attribute_filter_value( $vehicle ) ); ?>"
									<?php checked( $checked ); ?>
								>
								<span class="th-filter-vehicle__item-title">
									<?php echo esc_html__( 'Only exact fit parts for:', 'redparts-sputnik' ); ?>
								</span>
							</label>
						</li>
						<li class="th-filter-vehicle__vehicle">
							<div class="th-filter-vehicle__vehicle-title">
								<?php echo esc_html( Vehicles::get_vehicle_name( $vehicle ) ); ?>
							</div>
							<?php $description = Vehicles::get_vehicle_description( $vehicle ); ?>
							<?php if ( ! empty( $description ) ) : ?>
								<div class="th-filter-vehicle__vehicle-subtitle">
									<?php echo esc_html( $description ); ?>
								</div>
							<?php endif; ?>
						</li>
					</ul>
				<?php endif; ?>
				<div class="th-filter-vehicle__button">
					<button type="button" class="th-button th-button--style--secondary th-button--size--extra-small">
						<?php echo esc_html__( 'Select Vehicle', 'redparts-sputnik' ); ?>
					</button>
				</div>
			</div>
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
			);
		}
	}
}
