<?php
/**
 * Recent comments widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;
use WP_Comment;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Recent_Comments' ) ) {
	/**
	 * Class Recent_Comments
	 */
	class Recent_Comments extends Widget {
		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_recent_comments',
				'description'                 => esc_html_x( 'Displays the latest comments.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_recent_comments',
				esc_html_x( 'RedParts: Recent Comments', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$number = ( ! empty( $instance['number'] ) ) ? absint( $instance['number'] ) : 3;

			if ( ! $number ) {
				$number = 3;
			}

			$comments = get_comments(
				array(
					'number'      => $number,
					'status'      => 'approve',
					'post_status' => 'publish',
				)
			);

			if ( is_array( $comments ) && $comments ) :
				// Prime cache for associated posts. Prime post term cache if we need it for permalinks.
				$post_ids = array_unique( wp_list_pluck( $comments, 'comment_post_ID' ) );
				_prime_post_caches( $post_ids, strpos( get_option( 'permalink_structure' ), '%category%' ), false );

				?>
				<ul class="th-widget-comments__list">
					<?php foreach ( (array) $comments as $comment ) : ?>
						<?php /** Comment object. @var WP_Comment $comment */ ?>
						<li class="th-widget-comments__item">
							<div class="th-widget-comments__author">
								<a href="<?php echo esc_url( get_comment_link( $comment ) ); ?>">
									<?php echo esc_html( get_comment_author( $comment->comment_ID ) ); ?>
								</a>
							</div>
							<div class="th-widget-comments__content">
								<?php echo esc_html( get_comment_excerpt( $comment->comment_ID ) ); ?>
							</div>
							<div class="th-widget-comments__meta">
								<div class="th-widget-comments__date">
									<?php echo esc_html( get_comment_date( '', $comment->comment_ID ) ); ?>
								</div>
								<div class="th-widget-comments__name">
									<a
										href="<?php echo esc_url( get_post_permalink( $comment->comment_post_ID ) ); ?>"
										title="<?php echo esc_attr( get_the_title( $comment->comment_post_ID ) ); ?>">

										<?php echo esc_html( get_the_title( $comment->comment_post_ID ) ); ?>
									</a>
								</div>
							</div>
						</li>
					<?php endforeach; ?>
				</ul>
				<?php
			endif;
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
				array(
					'key'   => 'number',
					'label' => esc_html__( 'Number of comments to show', 'redparts-sputnik' ),
					'type'  => 'number',
				),
			);
		}
	}
}
