<?php
/**
 * Recent posts widget.
 *
 * @package RedParts\Sputnik
 * @since 1.0.0
 */

namespace RedParts\Sputnik\Widgets;

use RedParts\Sputnik\Widget;
use WP_Query;

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RedParts\Sputnik\Widgets\Recent_Posts' ) ) {
	/**
	 * Class Recent_Posts
	 */
	class Recent_Posts extends Widget {
		/**
		 * Constructor.
		 */
		public function __construct() {
			$widget_ops = array(
				'classname'                   => 'widget_redparts_sputnik_recent_posts',
				'description'                 => esc_html_x( 'Displays the latest posts.', 'Admin', 'redparts-sputnik' ),
				'customize_selective_refresh' => true,
			);

			parent::__construct(
				'redparts_sputnik_recent_posts',
				esc_html_x( 'RedParts: Recent Posts', 'Admin', 'redparts-sputnik' ),
				$widget_ops
			);
		}

		/**
		 * Echoes the widget body content.
		 *
		 * @param array $args     Display arguments including 'before_title', 'after_title', 'before_widget', and 'after_widget'.
		 * @param array $instance The settings for the particular instance of the widget.
		 */
		protected function widget_body( array $args, array $instance ) {
			$number = ( ! empty( $instance['number'] ) ) ? absint( $instance['number'] ) : 3;

			if ( ! $number ) {
				$number = 3;
			}

			$request = new WP_Query(
				array(
					'posts_per_page'      => $number,
					'no_found_rows'       => true,
					'post_status'         => 'publish',
					'ignore_sticky_posts' => true,
				)
			);

			if ( ! $request->have_posts() ) {
				return;
			}

			$thumbnail_size = apply_filters( 'redparts_sputnik_widget_recent_posts_image_size', 'thumbnail' );

			?>
			<div class="th-widget-posts__list">
				<?php foreach ( $request->posts as $post ) : ?>
					<?php
					$post_title = get_the_title( $post->ID );
					$title      = ( ! empty( $post_title ) ) ? $post_title : esc_html__( '(no title)', 'redparts-sputnik' );
					?>
					<div class="th-widget-posts__item">
						<?php if ( has_post_thumbnail( $post ) ) : ?>
							<div class="th-widget-posts__image">
								<a href="<?php the_permalink( $post->ID ); ?>">
									<?php echo get_the_post_thumbnail( $post, $thumbnail_size ); ?>
								</a>
							</div>
						<?php endif; ?>
						<div class="th-widget-posts__info">
							<div class="th-widget-posts__name">
								<a href="<?php the_permalink( $post->ID ); ?>">
									<?php echo esc_html( $title ); ?>
								</a>
							</div>
							<div class="th-widget-posts__date"><?php echo get_the_date( '', $post->ID ); ?></div>
						</div>
					</div>
				<?php endforeach; ?>
			</div>
			<?php
		}

		/**
		 * Returns form fields.
		 *
		 * @return array
		 */
		public function form_fields(): array {
			return array(
				array(
					'key'          => 'title',
					'label'        => esc_html__( 'Title', 'redparts-sputnik' ),
					'type'         => 'text',
					'translatable' => $this->is_elementor_widget(),
				),
				array(
					'key'   => 'number',
					'label' => esc_html__( 'Number of posts to show', 'redparts-sputnik' ),
					'type'  => 'number',
				),
			);
		}
	}
}
